use strict;

package Crosslinker::Scoring;
use lib 'lib';
use Crosslinker::Proteins;
use Crosslinker::Constants;
use Crosslinker::Config;
use base 'Exporter';
our @EXPORT = ('NextPermute', 'calc_score', 'find_ion_series');

######
#
# Functions for the Scoring of Peptides
#
#
######

sub _retry {
    my ($retrys, $func) = @_;
  attempt: {
        my $result;

        # if it works, return the result
        return $result if eval { $result = $func->(); 1 };

        # nah, it failed, if failure reason is not a lock, croak
        die $@ unless $@ =~ /database is locked/;

        # if we have 0 remaining retrys, stop trying.
        last attempt if $retrys < 1;

        # sleep for 0.1 seconds, and then try again.
        sleep 100 / $retrys;
        $retrys--;
        redo attempt;
    }

    die "Attempts Exceeded $@";
}

sub NextPermute(\@)    #From Algorithm::Loops
{

    my ($vals) = @_;
    my $last = $#{$vals};
    return "" if $last < 1;

    # Find last item not in reverse-sorted order:
    my $i = $last - 1;
    $i-- until $i < 0 || $vals->[$i] lt $vals->[ $i + 1 ];

    # If complete reverse sort, we are done!
    return "" if -1 == $i;

    # Re-sort the reversely-sorted tail of the list:
    @{$vals}[ $i + 1 .. $last ] = reverse @{$vals}[ $i + 1 .. $last ]
      if $vals->[ $i + 1 ] gt $vals->[$last];

    # Find next item that will make us "greater":
    my $j = $i + 1;
    $j++ until $vals->[$i] lt $vals->[$j];

    # Swap:
    @{$vals}[ $i, $j ] = @{$vals}[ $j, $i ];
    return 1;
}

sub calc_score {

    use Time::HiRes 'gettimeofday', 'tv_interval';
    use Math::BigRat;

    my (
        $protein_residuemass_ref, $MSn_string,      $d2_MSn_string,         $sequence,
        $modifications_ref,       $no_of_mods,      $modification,          $mass_of_hydrogen,
        $xlinker_mass,            $monolink_mass,   $seperation,            $reactive_site,
        $parent_charge,           $ms2_error,       $ms2_fragmentation_ref, $threshold,
        $no_xlink_at_cut_site,    $abundance_ratio, $fast_mode,		    $amber_codon,
    ) = @_;
    my %residuemass       = %{$protein_residuemass_ref};
    my $data              = $MSn_string;
    my $data2             = $d2_MSn_string;
    my %ms2_fragmentation = %{$ms2_fragmentation_ref};
    if (!defined $amber_codon) {$amber_codon = 0};

    # my $sequence  = $sequence;
    my $xlink = $xlinker_mass;
    my @masses = split "\n", $data;
    if (!defined $data2) { $data2 = '' }
    my @d2_masses = split "\n", $data2;
    my %ms2_masses;
    my $max_abundance    = 0;
    my $max_abundance_d2 = 0;
    my $total_ion_current_corrected;
    my $terminalmass  = 1.0078250 * 2 + 15.9949146 * 1;
    my $water         = 1.0078250 * 2 + 15.9949146 * 1;
    my $ammonia       = 1.0078250 * 3 + 14.003074;
    my %modifications = %{$modifications_ref};
    my $t0            = [ gettimeofday() ];
    my $best_x;
    my $best_y;
    my $best_alpha = 0;
    my $best_beta  = 0;

    # my $ms2_error = 0.5;  #Dalton error for assigin an ion to a pair of ions between the two spectra
    my $xlink_d   = $seperation;
    my $xlink_res = $reactive_site;
    my @xlink_pos;
    my $best_match = 0;

    if ($amber_codon == 1)  {
	      $xlink_res = 'Z';
	      $residuemass{'Z'} = $monolink_mass;
	      }

######
    #
    #Place ms2 data in database; need to grab this from the already existing database in the future.
    #
######

    my $dbh = DBI->connect("dbi:SQLite:dbname=:memory:", "", "", { RaiseError => 1, AutoCommit => 1 });
    my $masslist = $dbh->prepare("DROP TABLE IF EXISTS ms2;");
    _retry 15, sub { $masslist->execute() };
    my $TIC             = 0;
    my $no_of_ms2_peaks = 0;

    _retry 15, sub {
        $dbh->do(
"CREATE TABLE ms2  (mass REAL, abundance REAL, heavy_light REAL, possible_ion_shift REAL, possible_no_ion_shift REAL)");
    };
    my $newline = $dbh->prepare(
"INSERT INTO ms2 (mass , abundance,heavy_light, possible_ion_shift, possible_no_ion_shift ) VALUES (?, ?, ?, ?, ?)"
    );
    foreach my $mass_abundance (@masses) {
        $no_of_ms2_peaks = $no_of_ms2_peaks + 1;
        my ($mass, $abundance) = split " ", $mass_abundance;
        if ($abundance > $max_abundance) { $max_abundance = $abundance }
        $ms2_masses{$mass} = $abundance;
        _retry 15, sub { $newline->execute($mass, $abundance, 0, 0, 0) };
    }

    foreach my $mass_abundance (@d2_masses) {
        $no_of_ms2_peaks = $no_of_ms2_peaks + 1;
        my ($mass, $abundance) = split " ", $mass_abundance;
        if ($abundance > $max_abundance_d2) { $max_abundance_d2 = $abundance }
        $ms2_masses{$mass} = $abundance;
        _retry 15, sub { $newline->execute($mass, $abundance, 1, 0, 0) };
    }

    if ($seperation != 0) {
        my $ion_shift_matching = $dbh->prepare(
"UPDATE ms2 SET possible_no_ion_shift = 1  WHERE mass IN (SELECT ms2.mass as mass FROM ms2 INNER JOIN ms2 d2 ON
     (d2.mass between ms2.mass -  ? and ms2.mass + ?) WHERE ms2.heavy_light = 0 AND d2.heavy_light=1 )");

        # AND d2.abundance between ms2.abundance*0.5 and ms2.abundance*2

        _retry 15, sub { $ion_shift_matching->execute($ms2_error, $ms2_error) };

        $ion_shift_matching = $dbh->prepare(
"UPDATE ms2 SET possible_no_ion_shift = 1  WHERE mass IN (SELECT d2.mass as mass FROM ms2 INNER JOIN ms2 d2 ON
     (d2.mass between ms2.mass -  ? and ms2.mass + ?) WHERE ms2.heavy_light = 0 AND d2.heavy_light=1 )");

        _retry 15, sub { $ion_shift_matching->execute($ms2_error, $ms2_error) };

        #     warn "d0_Records_changed: $records_changed";
        my $max_intensity_ratio = 0.40;

        # $max_abundance_d2/$max_abundance

        if ($abundance_ratio == -1) { $abundance_ratio = $max_abundance_d2 / $max_abundance }
        for (my $charge = 1 ; $charge < ($parent_charge + 1) ; $charge++) {

            # warn "Charge - $charge,  Abundance:$abundance_ratio";
            $ion_shift_matching = $dbh->prepare(
"UPDATE ms2 SET possible_ion_shift = 1  WHERE mass IN (SELECT ms2.mass as mass FROM ms2 INNER JOIN ms2 d2 ON
     (d2.mass between ms2.mass -  ? and ms2.mass + ?) WHERE ms2.heavy_light = 0 AND d2.heavy_light=1  AND d2.abundance between ms2.abundance*(?)*? and ms2.abundance*(1/?)*?)"
            );
            _retry 15, sub {
                $ion_shift_matching->execute(
                                             $ms2_error - ($seperation / $charge),
                                             $ms2_error + ($seperation / $charge),
                                             $max_intensity_ratio,
                                             1 / $abundance_ratio,
                                             $max_intensity_ratio,
                                             1 / $abundance_ratio
                );
            };

            $ion_shift_matching = $dbh->prepare(
"UPDATE ms2 SET possible_ion_shift = 1  WHERE mass IN (SELECT d2.mass as mass FROM ms2 INNER JOIN ms2 d2 ON
     (d2.mass between ms2.mass -  ? and ms2.mass + ?) WHERE ms2.heavy_light = 0 AND d2.heavy_light=1 AND d2.abundance between ms2.abundance*(?)*? and ms2.abundance*(1/?)*? )"
            );

            _retry 15, sub {
                $ion_shift_matching->execute(
                                             $ms2_error - ($seperation / $charge),
                                             $ms2_error + ($seperation / $charge),
                                             $max_intensity_ratio,
                                             1 / $abundance_ratio,
                                             $max_intensity_ratio,
                                             1 / $abundance_ratio
                );
            };
        }

        # AND d2.abundance between ms2.abundance*0.5 and ms2.abundance*2

       #     warn "d2_Records_changed: $records_changed";
       #
       #       _retry 15, sub {$dbh->do("DELETE FROM ms2 WHERE possible_no_ion_shift = 0  AND possible_ion_shift = 0")};
    } else {
        $dbh->do("UPDATE ms2 SET possible_no_ion_shift = 1, possible_ion_shift = 1");
    }

######
    #
    #
    #
######

    my $highest_mz = (sort { $b <=> $a } keys %ms2_masses)[0];

######
    #
    # Perpare theoretical database;
    #
######

    $masslist = $dbh->prepare("DROP TABLE IF EXISTS theoretical;");
    _retry 15, sub {
        $dbh->do(
"CREATE TABLE theoretical  (mass REAL, ion_type, peptide_chain, position, x, y, sequence, charge,heavy_light, mod, crosslink_ion, not_crosslink_ion, is_scored)"
        );
    };
    my $new_theoretical = $dbh->prepare(
"INSERT INTO theoretical (mass, ion_type, peptide_chain, position, x, y, sequence, charge, heavy_light,mod, crosslink_ion,not_crosslink_ion, is_scored) VALUES (?, ?, ?,?, ? ,?, ?,?,?,?,?,?,?)"
    );
    _retry 15, sub { $dbh->do("CREATE INDEX theo_index ON theoretical (mass);") };

######
    #
    # Modify Correctly
    #
#######

    my @possible_modified_peptides;

    if ($modification ne 'NoMod' && $modification ne 'MonoLink') {
        $residuemass{'X'} =
          $residuemass{ $modifications{$modification}{Location} } + $modifications{$modification}{Delta};

        my $n        = $no_of_mods;
        my $M        = $modifications{$modification}{Location};
        my $X        = 'X';
        my @sections = split(/$M/, $sequence);

        my $size = @sections;

        my @replacements =
          sort split(//, ($X x $n) . ($M x ($size - $n - 1)));

        do {
            my @b = (@replacements, '');
            my $modified_peptide = join("", map { $sections[$_] . $b[$_] } (0 .. $#sections));    #. "\n";
            push @possible_modified_peptides, $modified_peptide;
        } while (NextPermute(@replacements));
    } else {
        push @possible_modified_peptides, $sequence;
    }

    my $best_tic_match;
    my $best_sequence;

    # my $best_x;
    # my $best_y;
    my @theoretical_ions;

    foreach (@possible_modified_peptides) {
        my $sequence = $_;

######
        #
        # Xlink position finder
        #
######
        my @peptides = split /-/, $sequence;

	my @xlink_residues = split (',', $xlink_res);


	if ($amber_codon == 0) {
        for (my $i = 0 ; $i < $#peptides + 1 ; $i++) {
            my $peptide = $peptides[$i];
            my @residues = split //, $peptide;
            my @tmp;
            for (my $n = 0 ; $n < @residues - $no_xlink_at_cut_site ; $n++) {


# 		warn $xlink_residues[$i], " ", $residues[$n], " ",$xlink_residues[$i] =~ /$residues[$n]/;
                if ( $xlink_residues[$i] =~ /$residues[$n]/) {
                    push @tmp, $n

                      #       last;
                }
            }
            $xlink_pos[$i] = [@tmp];
        }}
	else {
	  for (my $i = 0 ; $i < $#peptides + 1 ; $i++) {
            my $peptide = $peptides[$i];
            my @residues = split //, $peptide;
            my @tmp;
	    if ($i == 0) {
            for (my $n = 0 ; $n < @residues - $no_xlink_at_cut_site ; $n++) {
                    push @tmp, $n
            }
	    } else {
            for (my $n = 0 ; $n < @residues - $no_xlink_at_cut_site ; $n++) {
                if ($residues[$n] eq $xlink_res) {
                    push @tmp, $n; }}
	    }
            $xlink_pos[$i] = [@tmp];
	}}

        if ($sequence !~ /-/) { $xlink_pos[1] = [ (0) ] }
        ;    #Make sure monolinks actually search for position if $xlink_pos[1] was empty it would never loop.
        @peptides = split /-/, $sequence;
        my @xlink_half;
        for (my $i = 0 ; $i < $#peptides + 1 ; $i++) {
            my $peptide      = $peptides[$i];
            my @residues     = split //, $peptide;
            my $peptide_mass = 0;
            foreach my $residue (@residues) {    #split the peptide in indivual amino acids
                $peptide_mass =
                  $peptide_mass + $residuemass{$residue};    #tally the masses of each amino acid one at a time
            }
            $xlink_half[$i] = $peptide_mass + $terminalmass;
        }

        if ($#peptides + 1 == 1) { $xlink_half[1] = $terminalmass }

        for (my $x = 0 ; $x < scalar @{ $xlink_pos[0] } ; $x++) {
            for (my $y = 0 ; $y < scalar @{ $xlink_pos[1] } ; $y++) {
                my @xlink_position;
                $xlink_position[0] = $xlink_pos[0][$x];
                $xlink_position[1] = $xlink_pos[1][$y];

                my $matched_TIC;

                for (my $i = 0 ; $i < @peptides ; $i++) {
                    my $peptide = $peptides[$i];
                    my $mod_position = index($peptide, 'X');
######
                    #
                    # First Generate B and Y ion list
                    #
######
                    my $n = 0;

                    #B-ions
                    for (my $charge = 1 ; $charge < ($parent_charge + 1) ; $charge++) {
                        $n = 0;
                        my $add_isotope_to_heavy = 0;
                        my @residues             = split //, $peptide;
                        my $ion_mass             = 0;
                        my $residue_no           = 0;

                        foreach my $residue (@residues) {
                            $residue_no = $residue_no + 1;
                            $ion_mass   = $ion_mass + $residuemass{$residue};
                            if (   $residue_no == $xlink_position[$i] + 1
                                && $sequence =~ m/^[^-]*$/)    #not containing a '-'
                            {
                                $ion_mass             = $ion_mass + $monolink_mass;
                                $add_isotope_to_heavy = 1;
                            } elsif (   $residue_no > $xlink_position[$i]
                                     && $sequence !~ m/^[^-]*$/)
                            {
                                $n                    = 1;
                                $add_isotope_to_heavy = 1;
                            }

                            if (
                                $modification ne 'LoopLink'
                                || (
                                    ($xlink_position[$i] < $mod_position)
                                    && (   ($residue_no <= $xlink_position[$i])
                                        || ($residue_no > $mod_position))
                                )
                                || (
                                    ($xlink_position[$i] > $mod_position)
                                    && (   ($residue_no > $xlink_position[$i])
                                        || ($residue_no <= $mod_position))
                                )
                              )
                            {

                                # 			    warn "Generating B,$residue_no - $xlink_position[$i], $mod_position";
                                # 			    warn "Mono Link: $add_isotope_to_heavy,$residue_no, $xlink_position[$i] +1";
                                my $mz = (
                                          (
                                           $ion_mass +
                                             ($charge * $mass_of_hydrogen) +
                                             ($n * ($xlink + $xlink_half[ abs($i - 1) ]))
                                          ) / $charge
                                );
                                my $seperation_mz = ($add_isotope_to_heavy * ($seperation / $charge));
                                if ($ms2_fragmentation{'bions'}) {
                                    _retry 15, sub {
                                        $new_theoretical->execute(
                                                                  $mz,                   'B',
                                                                  $i,                    $residue_no,
                                                                  $xlink_position[0],    $xlink_position[1],
                                                                  $sequence,             $charge,
                                                                  0,                     '',
                                                                  $add_isotope_to_heavy, 1 - $add_isotope_to_heavy,
                                                                  $ms2_fragmentation{'bions-score'}
                                        );
                                    };    #need to calc position not needed atm.
                                    _retry 15, sub {
                                        $new_theoretical->execute(
                                            $mz + ($seperation_mz), 'B',
                                            $i,                    $residue_no,
                                            $xlink_position[0],    $xlink_position[1],
                                            $sequence,             $charge,
                                            1,                     '',
                                            $add_isotope_to_heavy, 1 - $add_isotope_to_heavy,
                                            $ms2_fragmentation{'bions-score'}
                                        );
                                    };    #heavy ions
                                }
                                if ($ms2_fragmentation{'aions'}) {
                                    _retry 15, sub {
                                        $new_theoretical->execute(
                                            $mz - ((26.9871 + 1.0078) / $charge), 'A',
                                            $i,                    $residue_no,
                                            $xlink_position[0],    $xlink_position[1],
                                            $sequence,             $charge,
                                            0,                     '',
                                            $add_isotope_to_heavy, 1 - $add_isotope_to_heavy,
                                            $ms2_fragmentation{'aions-score'}
                                        );
                                    };    #add A ion too
                                    _retry 15, sub {
                                        $new_theoretical->execute(
                                                                  $mz - (($seperation_mz + 26.9871 + 1.0078) / $charge),
                                                                  'A',
                                                                  $i,
                                                                  $residue_no,
                                                                  $xlink_position[0],
                                                                  $xlink_position[1],
                                                                  $sequence,
                                                                  $charge,
                                                                  1,
                                                                  '',
                                                                  $add_isotope_to_heavy,
                                                                  1 - $add_isotope_to_heavy,
                                                                  $ms2_fragmentation{'aions-score'}
                                        );
                                    };
                                }
                                my $fragment_residues = substr($peptides[$i], 0, $residue_no);
                                if ($n == 1 && @peptides > 1) {
                                    $fragment_residues = $fragment_residues . '-' . $peptides[ abs($i - 1) ];
                                }

               # 			    warn "n:$n i:$i +:$charge x:$xlink_position[0] y:$xlink_position[1] $fragment_residues  ", "\n";
                                if (   $fragment_residues =~ /[STED]/
                                    && $ms2_fragmentation{'waterloss'} == 1)
                                {    #WATER loss
                                    if ($ms2_fragmentation{'aions'}) {
                                        _retry 15, sub {
                                            $new_theoretical->execute(
                                                $mz - ($water / $charge), 'A',
                                                $i,                    $residue_no,
                                                $xlink_position[0],    $xlink_position[1],
                                                $sequence,             $charge,
                                                0,                     '[-H2O]',
                                                $add_isotope_to_heavy, 1 - $add_isotope_to_heavy,
                                                $ms2_fragmentation{'waterloss-score'}
                                            );
                                        };
                                        _retry 15, sub {
                                            $new_theoretical->execute(
                                                                      $mz - ((26.9871 + 1.0078 - $water) / $charge),
                                                                      'A',
                                                                      $i,
                                                                      $residue_no,
                                                                      $xlink_position[0],
                                                                      $xlink_position[1],
                                                                      $sequence,
                                                                      $charge,
                                                                      0,
                                                                      '[-H2O]',
                                                                      $add_isotope_to_heavy,
                                                                      1 - $add_isotope_to_heavy,
                                                                      $ms2_fragmentation{'waterloss-score'}
                                            );
                                        };
                                    }
                                    if ($ms2_fragmentation{'bions'}) {
                                        _retry 15, sub {
                                            $new_theoretical->execute(
                                                                      $mz + ($seperation_mz - ($water / $charge)),
                                                                      'B',
                                                                      $i,
                                                                      $residue_no,
                                                                      $xlink_position[0],
                                                                      $xlink_position[1],
                                                                      $sequence,
                                                                      $charge,
                                                                      1,
                                                                      '[-H2O]',
                                                                      $add_isotope_to_heavy,
                                                                      1 - $add_isotope_to_heavy,
                                                                      $ms2_fragmentation{'waterloss-score'}
                                            );
                                        };
                                        _retry 15, sub {
                                            $new_theoretical->execute(
                                                $mz - ($seperation_mz + (+26.9871 + 1.0078 - $water) / $charge),
                                                'B', $i, $residue_no, $xlink_position[0], $xlink_position[1], $sequence,
                                                $charge,               1, '[-H2O]',
                                                $add_isotope_to_heavy, 1 - $add_isotope_to_heavy,
                                                $ms2_fragmentation{'waterloss-score'}
                                            );
                                        };
                                    }
                                }
                                if (   $fragment_residues =~ /[NQKR]/
                                    && $ms2_fragmentation{'ammonialoss'})
                                {    #Ammonia loss
                                    if ($ms2_fragmentation{'aions'}) {
                                        _retry 15, sub {
                                            $new_theoretical->execute(
                                                $mz - ($ammonia / $charge), 'A',
                                                $i,                    $residue_no,
                                                $xlink_position[0],    $xlink_position[1],
                                                $sequence,             $charge,
                                                0,                     '[-NH3]',
                                                $add_isotope_to_heavy, 1 - $add_isotope_to_heavy,
                                                $ms2_fragmentation{'ammonialoss-score'}
                                            );
                                        };
                                        _retry 15, sub {
                                            $new_theoretical->execute(
                                                                      $mz - ((26.9871 + 1.0078 - $ammonia) / $charge),
                                                                      'A',
                                                                      $i,
                                                                      $residue_no,
                                                                      $xlink_position[0],
                                                                      $xlink_position[1],
                                                                      $sequence,
                                                                      $charge,
                                                                      0,
                                                                      '[-NH3]',
                                                                      $add_isotope_to_heavy,
                                                                      1 - $add_isotope_to_heavy,
                                                                      $ms2_fragmentation{'ammonialoss-score'}
                                            );
                                        };
                                    }
                                    if ($ms2_fragmentation{'bions'}) {
                                        _retry 15, sub {
                                            $new_theoretical->execute(
                                                                      $mz + ($seperation_mz - ($ammonia / $charge)),
                                                                      'B',
                                                                      $i,
                                                                      $residue_no,
                                                                      $xlink_position[0],
                                                                      $xlink_position[1],
                                                                      $sequence,
                                                                      $charge,
                                                                      1,
                                                                      '[-NH3]',
                                                                      $add_isotope_to_heavy,
                                                                      1 - $add_isotope_to_heavy,
                                                                      $ms2_fragmentation{'ammonialoss-score'}
                                            );
                                        };
                                        _retry 15, sub {
                                            $new_theoretical->execute(
                                                $mz - ($seperation_mz + (+26.9871 + 1.0078 - $ammonia) / $charge),
                                                'B', $i, $residue_no, $xlink_position[0], $xlink_position[1], $sequence,
                                                $charge,               1, '[-NH3]',
                                                $add_isotope_to_heavy, 1 - $add_isotope_to_heavy,
                                                $ms2_fragmentation{'ammonialoss-score'}
                                            );
                                        };
                                    }
                                }

                            }
                        }
                    }

####  Y-ions
                    my @residues = split //, $peptide;
                    if ($ms2_fragmentation{'yions'}) {
                        for (my $charge = 1 ; $charge < $parent_charge + 1 ; $charge++) {
                            my $add_isotope_to_heavy = 0;
                            if ($sequence !~ m/^[^-]*$/) {
                                $n = 1;
                            }
                            my @residues = split //, $peptide;
                            my $peptide_mass = 0;

                            foreach my $residue (@residues) {    #split the peptide in indivual amino acids
                                $peptide_mass =
                                  $peptide_mass +
                                  $residuemass{$residue};        #tally the masses of each amino acid one at a time
                            }
                            if ($sequence =~ m/^[^-]*$/) {
                                $peptide_mass = $peptide_mass + $monolink_mass;
                            }
                            my $ion_mass   = $peptide_mass;
                            my $residue_no = 0;

                            foreach my $residue (@residues) {
                                $residue_no           = $residue_no + 1;
                                $add_isotope_to_heavy = 1;
                                if ($residue_no == @residues) { last; }
                                $ion_mass = $ion_mass - $residuemass{$residue};
                                if (   $residue_no == $xlink_position[$i] + 1
                                    && $sequence =~ m/^[^-]*$/)
                                {
                                    $ion_mass             = $ion_mass - $monolink_mass;
                                    $add_isotope_to_heavy = 1;
                                } elsif (   $residue_no > $xlink_position[$i]
                                         && $sequence !~ m/^[^-]*$/)
                                {
                                    $n                    = 0;
                                    $add_isotope_to_heavy = 0;
                                } elsif (   $residue_no > $xlink_position[$i]
                                         && $sequence =~ m/^[^-]*$/)
                                {
                                    $add_isotope_to_heavy = 0;
                                }

                                if (
                                    $modification ne 'LoopLink'
                                    || (
                                        ($xlink_position[$i] < $mod_position)
                                        && (   ($residue_no <= $xlink_position[$i])
                                            || ($residue_no > $mod_position))
                                    )
                                    || (
                                        ($xlink_position[$i] > $mod_position)
                                        && (   ($residue_no > $xlink_position[$i])
                                            || ($residue_no <= $mod_position))
                                    )
                                  )
                                {

             #                             				    warn "Generating Y,$residue_no - $xlink_position[$i], $mod_position";

                                    my $mz = (
                                              (
                                               $ion_mass +
                                                 $terminalmass +
                                                 ($charge * $mass_of_hydrogen) +
                                                 ($n * ($xlink + $xlink_half[ abs($i - 1) ]))
                                              ) / $charge
                                    );
                                    my $seperation_mz = ($add_isotope_to_heavy * ($seperation / $charge));
                                    my $fragment_residues = substr($peptides[$i], $residue_no);
                                    if ($n == 1 && @peptides > 1) {
                                        $fragment_residues = $fragment_residues . '-' . $peptides[ abs($i - 1) ];
                                    }

#                             			    warn "n:$n i:$i +:$charge x:$xlink_position[0] y:$xlink_position[1] $fragment_residues  ", "\n";
#                                                            warn $mz, " Y $i $residue_no";
                                    _retry 15, sub {
                                        $new_theoretical->execute(
                                                                  $mz,                   'Y',
                                                                  $i,                    @residues - $residue_no,
                                                                  $xlink_position[0],    $xlink_position[1],
                                                                  $sequence,             $charge,
                                                                  0,                     '',
                                                                  $add_isotope_to_heavy, 1 - $add_isotope_to_heavy,
                                                                  $ms2_fragmentation{'yions-score'}
                                        );
                                    };    #need to calc position not needed atm.

                                    #                                                             warn "Y $i", $mz;
                                    _retry 15, sub {
                                        $new_theoretical->execute(
                                            $mz + ($seperation_mz), 'Y',
                                            $i,                    @residues - $residue_no,
                                            $xlink_position[0],    $xlink_position[1],
                                            $sequence,             $charge,
                                            1,                     '',
                                            $add_isotope_to_heavy, 1 - $add_isotope_to_heavy,
                                            $ms2_fragmentation{'yions-score'}
                                        );
                                    };    #need to calc position not needed atm.

                                    #                            				warn  "Y(H) $i", $mz+$seperation_mz ;
                                    if (   $fragment_residues =~ /[STED]/
                                        && $ms2_fragmentation{'waterloss'})
                                    {     #WATER loss
                                        _retry 15, sub {
                                            $new_theoretical->execute(
                                                $mz - ($water / $charge), 'Y',
                                                $i,                    @residues - $residue_no,
                                                $xlink_position[0],    $xlink_position[1],
                                                $sequence,             $charge,
                                                0,                     '[-H2O]',
                                                $add_isotope_to_heavy, 1 - $add_isotope_to_heavy,
                                                $ms2_fragmentation{'waterloss-score'}
                                            );
                                        };    #need to calc position not needed atm.
                                        _retry 15, sub {
                                            $new_theoretical->execute(
                                                                      $mz + ($seperation_mz) - ($water / $charge),
                                                                      'Y',
                                                                      $i,
                                                                      @residues - $residue_no,
                                                                      $xlink_position[0],
                                                                      $xlink_position[1],
                                                                      $sequence,
                                                                      $charge,
                                                                      1,
                                                                      '[-NH3]',
                                                                      $add_isotope_to_heavy,
                                                                      1 - $add_isotope_to_heavy,
                                                                      $ms2_fragmentation{'waterloss-score'}
                                            );
                                        };    #need to calc position not needed atm.
                                    }
                                    if (   $fragment_residues =~ /[NQKR]/
                                        && $ms2_fragmentation{'ammonialoss'})
                                    {         #Ammonia loss
                                        _retry 15, sub {
                                            $new_theoretical->execute(
                                                $mz - ($ammonia / $charge), 'Y',
                                                $i,                    @residues - $residue_no,
                                                $xlink_position[0],    $xlink_position[1],
                                                $sequence,             $charge,
                                                0,                     '[-H2O]',
                                                $add_isotope_to_heavy, 1 - $add_isotope_to_heavy,
                                                $ms2_fragmentation{'ammonialoss-score'}
                                            );
                                        };    #need to calc position not needed atm.
                                        _retry 15, sub {
                                            $new_theoretical->execute(
                                                                      $mz + ($seperation_mz) - ($ammonia / $charge),
                                                                      'Y',
                                                                      $i,
                                                                      @residues - $residue_no,
                                                                      $xlink_position[0],
                                                                      $xlink_position[1],
                                                                      $sequence,
                                                                      $charge,
                                                                      1,
                                                                      '[-NH3]',
                                                                      $add_isotope_to_heavy,
                                                                      1 - $add_isotope_to_heavy,
                                                                      $ms2_fragmentation{'ammonialoss-score'}
                                            );
                                        };    #need to calc position not needed atm.
                                    }
                                }
                            }
                        }
                    }
                }

###
                #
                # Match ions
                #
                #
####

                my $q_max          = 5;
                my $interval_range = 100;
                my $n              = 0;
                my $k;
                my $score_total          = 1;
                my $score_alpha_total    = 1;
                my $score_beta_total     = 1;
                my $score_alpha_d2_total = 1;
                my $score_beta_d2_total  = 1;
                my $d0_score_total       = 1;
                my $d2_score_total       = 1;

                my @pascal_row1  = qw(1 1);
                my @pascal_row2  = qw(1 2 1);
                my @pascal_row3  = qw(1 3 3 1);
                my @pascal_row4  = qw(1 4 6 4 1);
                my @pascal_row5  = qw(1 5 10 10 5 1);
                my @pascal_row6  = qw(1 6 15 20 15 6 1);
                my @pascal_row7  = qw(1 7 21 35 35 21 7 1);
                my @pascal_row8  = qw(1 8 28 56 70 56 28 8 1);
                my @pascal_row9  = qw(1 9 36 84 126 126 84 36 9 1);
                my @pascal_row10 = qw(1 10 45 120 210 252 210 120 45 10 1);
                my @pascal_row11 = qw(1 11 55 165 330 462 462 330 165 55 11 1);
                my @pascal_row12 = qw(1 12 66 220 495 792 924 792 495 220 66 12 1);
                my @pascal_row13 = qw(1 13 78 286 715 1287 1716 1716 1287 715 286 78 13 1);
                my @pascal_row14 = qw(1 14 91 364 1001 2002 3003 3432 3003 2002 1001 364 91 14 1);
                my @pascal_row15 = qw(1 15 105 455 1365 3003 5005 6435 6435 5005 3003 1365 455 105 15 1);
                my @pascal_row16 = qw(1 16 120 560 1820 4368 8008 11440 12870 11440 8008 4368 1820 560 120 16 1);
                my @pascal_row17 =
                  qw(1 17 136 680 2380 6188 12376 19448 24310 24310 19448 12376 6188 2380 680 136 17 1);
                my @pascal_row18 =
                  qw(1 18 153 816 3060 8568 18564 31824 43758 48620 43758 31824 18564 8568 3060 816 153 18 1);
                my @pascal_row19 =
                  qw(1 19 171 969 3876 11628 27132 50388 75582 92378 92378 75582 50388 27132 11628 3876 969 171 19 1);
                my @pascal_row20 =
                  qw(1 20 190 1140 4845 15504 38760 77520 125970 167960 184756 167960 125970 77520 38760 15504 4845 1140 190 20 1);
                my @pascal_row21 =
                  qw(1 21 210 1330 5985 20349 54264 116280 203490 293930 352716 352716 293930 203490 116280 54264 20349 5985 1330 210 21 1);
                my @pascal_row22 =
                  qw(1 22 231 1540 7315 26334 74613 170544 319770 497420 646646 705432 646646 497420 319770 170544 74613 26334 7315 1540 231 22 1);
                my @pascal_row23 =
                  qw(1 23 253 1771 8855 33649 100947 245157 490314 817190 1144066 1352078 1352078 1144066 817190 490314 245157 100947 33649 8855 1771 253 23 1);
                my @pascal_row24 =
                  qw(1 24 276 2024 10626 42504 134596 346104 735471 1307504 1961256 2496144 2704156 2496144 1961256 1307504 735471 346104 134596 42504 10626 2024 276 24 1);
                my @pascal_row25 =
                  qw(1 25 300 2300 12650 53130 177100 480700 1081575 2042975 3268760 4457400 5200300 5200300 4457400 3268760 2042975 1081575 480700 177100 53130 12650 2300 300 25 1);
                my @pascal_row26 =
                  qw(1 26 325 2600 14950 65780 230230 657800 1562275 3124550 5311735 7726160 9657700 10400600 9657700 7726160 5311735 3124550 1562275 657800 230230 65780 14950 2600 325 26 1);
                my @pascal_row27 =
                  qw(1 27 351 2925 17550 80730 296010 888030 2220075 4686825 8436285 13037895 17383860 20058300 20058300 17383860 13037895 8436285 4686825 2220075 888030 296010 80730 17550 2925 351 27 1);
                my @pascal_row28 =
                  qw(1 28 378 3276 20475 98280 376740 1184040 3108105 6906900 13123110 21474180 30421755 37442160 40116600 37442160 30421755 21474180 13123110 6906900 3108105 1184040 376740 98280 20475 3276 378 28 1);
                my @pascal_row29 =
                  qw(1 29 406 3654 23751 118755 475020 1560780 4292145 10015005 20030010 34597290 51895935 67863915 77558760 77558760 67863915 51895935 34597290 20030010 10015005 4292145 1560780 475020 118755 23751 3654 406 29 1);
                my @pascal_row30 =
                  qw(1 30 435 4060 27405 142506 593775 2035800 5852925 14307150 30045015 54627300 86493225 119759850 145422675 155117520 145422675 119759850 86493225 54627300 30045015 14307150 5852925 2035800 593775 142506 27405 4060 435 30 1);
                my @pascal_row31 =
                  qw(1 31 465 4495 31465 169911 736281 2629575 7888725 20160075 44352165 84672315 141120525 206253075 265182525 300540195 300540195 265182525 206253075 141120525 84672315 44352165 20160075 7888725 2629575 736281 169911 31465 4495 465 31 1);
                my @pascal_row32 =
                  qw(1 32 496 4960 35960 201376 906192 3365856 10518300 28048800 64512240 129024480 225792840 347373600 471435600 565722720 601080390 565722720 471435600 347373600 225792840 129024480 64512240 28048800 10518300 3365856 906192 201376 35960 4960 496 32 1);
                my @pascal_row33 =
                  qw(1 33 528 5456 40920 237336 1107568 4272048 13884156 38567100 92561040 193536720 354817320 573166440 818809200 1037158320 1166803110 1166803110 1037158320 818809200 573166440 354817320 193536720 92561040 38567100 13884156 4272048 1107568 237336 40920 5456 528 33 1);
                my @pascal_row34 =
                  qw(1 34 561 5984 46376 278256 1344904 5379616 18156204 52451256 131128140 286097760 548354040 927983760 1391975640 1855967520 2203961430 2333606220 2203961430 1855967520 1391975640 927983760 548354040 286097760 131128140 52451256 18156204 5379616 1344904 278256 46376 5984 561 34 1);
                my @pascal_row35 =
                  qw(1 35 595 6545 52360 324632 1623160 6724520 23535820 70607460 183579396 417225900 834451800 1476337800 2319959400 3247943160 4059928950 4537567650 4537567650 4059928950 3247943160 2319959400 1476337800 834451800 417225900 183579396 70607460 23535820 6724520 1623160 324632 52360 6545 595 35 1);
                my @pascal_row36 =
                  qw(1 36 630 7140 58905 376992 1947792 8347680 30260340 94143280 254186856 600805296 1251677700 2310789600 3796297200 5567902560 7307872110 8597496600 9075135300 8597496600 7307872110 5567902560 3796297200 2310789600 1251677700 600805296 254186856 94143280 30260340 8347680 1947792 376992 58905 7140 630 36 1);
                my @pascal_row37 =
                  qw(1 37 666 7770 66045 435897 2324784 10295472 38608020 124403620 348330136 854992152 1852482996 3562467300 6107086800 9364199760 12875774670 15905368710 17672631900 17672631900 15905368710 12875774670 9364199760 6107086800 3562467300 1852482996 854992152 348330136 124403620 38608020 10295472 2324784 435897 66045 7770 666 37 1);
                my @pascal_row38 =
                  qw(1 38 703 8436 73815 501942 2760681 12620256 48903492 163011640 472733756 1203322288 2707475148 5414950296 9669554100 15471286560 22239974430 28781143380 33578000610 35345263800 33578000610 28781143380 22239974430 15471286560 9669554100 5414950296 2707475148 1203322288 472733756 163011640 48903492 12620256 2760681 501942 73815 8436 703 38 1);
                my @pascal_row39 =
                  qw(1 39 741 9139 82251 575757 3262623 15380937 61523748 211915132 635745396 1676056044 3910797436 8122425444 15084504396 25140840660 37711260990 51021117810 62359143990 68923264410 68923264410 62359143990 51021117810 37711260990 25140840660 15084504396 8122425444 3910797436 1676056044 635745396 211915132 61523748 15380937 3262623 575757 82251 9139 741 39 1);
                my @pascal_row40 =
                  qw(1 40 780 9880 91390 658008 3838380 18643560 76904685 273438880 847660528 2311801440 5586853480 12033222880 23206929840 40225345056 62852101650 88732378800 113380261800 131282408400 137846528820 131282408400 113380261800 88732378800 62852101650 40225345056 23206929840 12033222880 5586853480 2311801440 847660528 273438880 76904685 18643560 3838380 658008 91390 9880 780 40 1);
                my @pascal_row41 =
                  qw(1 41 820 10660 101270 749398 4496388 22481940 95548245 350343565 1121099408 3159461968 7898654920 17620076360 35240152720 63432274896 103077446706 151584480450 202112640600 244662670200 269128937220 269128937220 244662670200 202112640600 151584480450 103077446706 63432274896 35240152720 17620076360 7898654920 3159461968 1121099408 350343565 95548245 22481940 4496388 749398 101270 10660 820 41 1);
                my @pascal_row42 =
                  qw(1 42 861 11480 111930 850668 5245786 26978328 118030185 445891810 1471442973 4280561376 11058116888 25518731280 52860229080 98672427616 166509721602 254661927156 353697121050 446775310800 513791607420 538257874440 513791607420 446775310800 353697121050 254661927156 166509721602 98672427616 52860229080 25518731280 11058116888 4280561376 1471442973 445891810 118030185 26978328 5245786 850668 111930 11480 861 42 1);
                my @pascal_row43 =
                  qw(1 43 903 12341 123410 962598 6096454 32224114 145008513 563921995 1917334783 5752004349 15338678264 36576848168 78378960360 151532656696 265182149218 421171648758 608359048206 800472431850 960566918220 1052049481860 1052049481860 960566918220 800472431850 608359048206 421171648758 265182149218 151532656696 78378960360 36576848168 15338678264 5752004349 1917334783 563921995 145008513 32224114 6096454 962598 123410 12341 903 43 1);
                my @pascal_row44 =
                  qw(1 44 946 13244 135751 1086008 7059052 38320568 177232627 708930508 2481256778 7669339132 21090682613 51915526432 114955808528 229911617056 416714805914 686353797976 1029530696964 1408831480056 1761039350070 2012616400080 2104098963720 2012616400080 1761039350070 1408831480056 1029530696964 686353797976 416714805914 229911617056 114955808528 51915526432 21090682613 7669339132 2481256778 708930508 177232627 38320568 7059052 1086008 135751 13244 946 44 1);
                my @pascal_row45 =
                  qw(1 45 990 14190 148995 1221759 8145060 45379620 215553195 886163135 3190187286 10150595910 28760021745 73006209045 166871334960 344867425584 646626422970 1103068603890 1715884494940 2438362177020 3169870830126 3773655750150 4116715363800 4116715363800 3773655750150 3169870830126 2438362177020 1715884494940 1103068603890 646626422970 344867425584 166871334960 73006209045 28760021745 10150595910 3190187286 886163135 215553195 45379620 8145060 1221759 148995 14190 990 45 1);
                my @pascal_row46 =
                  qw(1 46 1035 15180 163185 1370754 9366819 53524680 260932815 1101716330 4076350421 13340783196 38910617655 101766230790 239877544005 511738760544 991493848554 1749695026860 2818953098830 4154246671960 5608233007146 6943526580276 7890371113950 8233430727600 7890371113950 6943526580276 5608233007146 4154246671960 2818953098830 1749695026860 991493848554 511738760544 239877544005 101766230790 38910617655 13340783196 4076350421 1101716330 260932815 53524680 9366819 1370754 163185 15180 1035 46 1);
                my @pascal_row47 =
                  qw(1 47 1081 16215 178365 1533939 10737573 62891499 314457495 1362649145 5178066751 17417133617 52251400851 140676848445 341643774795 751616304549 1503232609098 2741188875414 4568648125690 6973199770790 9762479679106 12551759587422 14833897694226 16123801841550 16123801841550 14833897694226 12551759587422 9762479679106 6973199770790 4568648125690 2741188875414 1503232609098 751616304549 341643774795 140676848445 52251400851 17417133617 5178066751 1362649145 314457495 62891499 10737573 1533939 178365 16215 1081 47 1);
                my @pascal_row48 =
                  qw(1 48 1128 17296 194580 1712304 12271512 73629072 377348994 1677106640 6540715896 22595200368 69668534468 192928249296 482320623240 1093260079344 2254848913647 4244421484512 7309837001104 11541847896480 16735679449896 22314239266528 27385657281648 30957699535776 32247603683100 30957699535776 27385657281648 22314239266528 16735679449896 11541847896480 7309837001104 4244421484512 2254848913647 1093260079344 482320623240 192928249296 69668534468 22595200368 6540715896 1677106640 377348994 73629072 12271512 1712304 194580 17296 1128 48 1);
                my @pascal_row49 =
                  qw(1 49 1176 18424 211876 1906884 13983816 85900584 450978066 2054455634 8217822536 29135916264 92263734836 262596783764 675248872536 1575580702584 3348108992991 6499270398159 11554258485616 18851684897584 28277527346376 39049918716424 49699896548176 58343356817424 63205303218876 63205303218876 58343356817424 49699896548176 39049918716424 28277527346376 18851684897584 11554258485616 6499270398159 3348108992991 1575580702584 675248872536 262596783764 92263734836 29135916264 8217822536 2054455634 450978066 85900584 13983816 1906884 211876 18424 1176 49 1);
                my @pascal_row50 =
                  qw(1 50 1225 19600 230300 2118760 15890700 99884400 536878650 2505433700 10272278170 37353738800 121399651100 354860518600 937845656300 2250829575120 4923689695575 9847379391150 18053528883775 30405943383200 47129212243960 67327446062800 88749815264600 108043253365600 121548660036300 126410606437752 121548660036300 108043253365600 88749815264600 67327446062800 47129212243960 30405943383200 18053528883775 9847379391150 4923689695575 2250829575120 937845656300 354860518600 121399651100 37353738800 10272278170 2505433700 536878650 99884400 15890700 2118760 230300 19600 1225 50 1);
                my @pascal_row51 =
                  qw(1 51 1275 20825 249900 2349060 18009460 115775100 636763050 3042312350 12777711870 47626016970 158753389900 476260169700 1292706174900 3188675231420 7174519270695 14771069086725 27900908274925 48459472266975 77535155627160 114456658306760 156077261327400 196793068630200 229591913401900 247959266474052 247959266474052 229591913401900 196793068630200 156077261327400 114456658306760 77535155627160 48459472266975 27900908274925 14771069086725 7174519270695 3188675231420 1292706174900 476260169700 158753389900 47626016970 12777711870 3042312350 636763050 115775100 18009460 2349060 249900 20825 1275 51 1);
                my @pascal_row52 =
                  qw(1 52 1326 22100 270725 2598960 20358520 133784560 752538150 3679075400 15820024220 60403728840 206379406870 635013559600 1768966344600 4481381406320 10363194502115 21945588357420 42671977361650 76360380541900 125994627894135 191991813933920 270533919634160 352870329957600 426384982032100 477551179875952 495918532948104 477551179875952 426384982032100 352870329957600 270533919634160 191991813933920 125994627894135 76360380541900 42671977361650 21945588357420 10363194502115 4481381406320 1768966344600 635013559600 206379406870 60403728840 15820024220 3679075400 752538150 133784560 20358520 2598960 270725 22100 1326 52 1);
                my @pascal_row53 =
                  qw(1 53 1378 23426 292825 2869685 22957480 154143080 886322710 4431613550 19499099620 76223753060 266783135710 841392966470 2403979904200 6250347750920 14844575908435 32308782859535 64617565719070 119032357903550 202355008436035 317986441828055 462525733568080 623404249591760 779255311989700 903936161908052 973469712824056 973469712824056 903936161908052 779255311989700 623404249591760 462525733568080 317986441828055 202355008436035 119032357903550 64617565719070 32308782859535 14844575908435 6250347750920 2403979904200 841392966470 266783135710 76223753060 19499099620 4431613550 886322710 154143080 22957480 2869685 292825 23426 1378 53 1);
                my @pascal_row54 =
                  qw(1 54 1431 24804 316251 3162510 25827165 177100560 1040465790 5317936260 23930713170 95722852680 343006888770 1108176102180 3245372870670 8654327655120 21094923659355 47153358767970 96926348578605 183649923622620 321387366339585 520341450264090 780512175396135 1085929983159840 1402659561581460 1683191473897752 1877405874732108 1946939425648112 1877405874732108 1683191473897752 1402659561581460 1085929983159840 780512175396135 520341450264090 321387366339585 183649923622620 96926348578605 47153358767970 21094923659355 8654327655120 3245372870670 1108176102180 343006888770 95722852680 23930713170 5317936260 1040465790 177100560 25827165 3162510 316251 24804 1431 54 1);
                my @pascal_row55 =
                  qw(1 55 1485 26235 341055 3478761 28989675 202927725 1217566350 6358402050 29248649430 119653565850 438729741450 1451182990950 4353548972850 11899700525790 29749251314475 68248282427325 144079707346575 280576272201225 505037289962205 841728816603675 1300853625660225 1866442158555975 2488589544741300 3085851035479212 3560597348629860 3824345300380220 3824345300380220 3560597348629860 3085851035479212 2488589544741300 1866442158555975 1300853625660225 841728816603675 505037289962205 280576272201225 144079707346575 68248282427325 29749251314475 11899700525790 4353548972850 1451182990950 438729741450 119653565850 29248649430 6358402050 1217566350 202927725 28989675 3478761 341055 26235 1485 55 1);
                my @pascal_row56 =
                  qw(1 56 1540 27720 367290 3819816 32468436 231917400 1420494075 7575968400 35607051480 148902215280 558383307300 1889912732400 5804731963800 16253249498640 41648951840265 97997533741800 212327989773900 424655979547800 785613562163430 1346766106565880 2142582442263900 3167295784216200 4355031703297275 5574440580220512 6646448384109072 7384942649010080 7648690600760440 7384942649010080 6646448384109072 5574440580220512 4355031703297275 3167295784216200 2142582442263900 1346766106565880 785613562163430 424655979547800 212327989773900 97997533741800 41648951840265 16253249498640 5804731963800 1889912732400 558383307300 148902215280 35607051480 7575968400 1420494075 231917400 32468436 3819816 367290 27720 1540 56 1);
                my @pascal_row57 =
                  qw(1 57 1596 29260 395010 4187106 36288252 264385836 1652411475 8996462475 43183019880 184509266760 707285522580 2448296039700 7694644696200 22057981462440 57902201338905 139646485582065 310325523515700 636983969321700 1210269541711230 2132379668729310 3489348548829780 5309878226480100 7522327487513475 9929472283517787 12220888964329584 14031391033119152 15033633249770520 15033633249770520 14031391033119152 12220888964329584 9929472283517787 7522327487513475 5309878226480100 3489348548829780 2132379668729310 1210269541711230 636983969321700 310325523515700 139646485582065 57902201338905 22057981462440 7694644696200 2448296039700 707285522580 184509266760 43183019880 8996462475 1652411475 264385836 36288252 4187106 395010 29260 1596 57 1);
                my @pascal_row58 =
                  qw(1 58 1653 30856 424270 4582116 40475358 300674088 1916797311 10648873950 52179482355 227692286640 891794789340 3155581562280 10142940735900 29752626158640 79960182801345 197548686920970 449972009097765 947309492837400 1847253511032930 3342649210440540 5621728217559090 8799226775309880 12832205713993575 17451799771031262 22150361247847371 26252279997448736 29065024282889672 30067266499541040 29065024282889672 26252279997448736 22150361247847371 17451799771031262 12832205713993575 8799226775309880 5621728217559090 3342649210440540 1847253511032930 947309492837400 449972009097765 197548686920970 79960182801345 29752626158640 10142940735900 3155581562280 891794789340 227692286640 52179482355 10648873950 1916797311 300674088 40475358 4582116 424270 30856 1653 58 1);
                my @pascal_row59 =
                  qw(1 59 1711 32509 455126 5006386 45057474 341149446 2217471399 12565671261 62828356305 279871768995 1119487075980 4047376351620 13298522298180 39895566894540 109712808959985 277508869722315 647520696018735 1397281501935165 2794563003870330 5189902721473470 8964377427999630 14420954992868970 21631432489303455 30284005485024837 39602161018878633 48402641245296107 55317304280338408 59132290782430712 59132290782430712 55317304280338408 48402641245296107 39602161018878633 30284005485024837 21631432489303455 14420954992868970 8964377427999630 5189902721473470 2794563003870330 1397281501935165 647520696018735 277508869722315 109712808959985 39895566894540 13298522298180 4047376351620 1119487075980 279871768995 62828356305 12565671261 2217471399 341149446 45057474 5006386 455126 32509 1711 59 1);
                my @pascal_row60 =
                  qw(1 60 1770 34220 487635 5461512 50063860 386206920 2558620845 14783142660 75394027566 342700125300 1399358844975 5166863427600 17345898649800 53194089192720 149608375854525 387221678682300 925029565741050 2044802197953900 4191844505805495 7984465725343800 14154280149473100 23385332420868600 36052387482172425 51915437974328292 69886166503903470 88004802264174740 103719945525634515 114449595062769120 118264581564861424 114449595062769120 103719945525634515 88004802264174740 69886166503903470 51915437974328292 36052387482172425 23385332420868600 14154280149473100 7984465725343800 4191844505805495 2044802197953900 925029565741050 387221678682300 149608375854525 53194089192720 17345898649800 5166863427600 1399358844975 342700125300 75394027566 14783142660 2558620845 386206920 50063860 5461512 487635 34220 1770 60 1);
                my @pascal_row61 =
                  qw(1 61 1830 35990 521855 5949147 55525372 436270780 2944827765 17341763505 90177170226 418094152866 1742058970275 6566222272575 22512762077400 70539987842520 202802465047245 536830054536825 1312251244423350 2969831763694950 6236646703759395 12176310231149295 22138745874816900 37539612570341700 59437719903041025 87967825456500717 121801604478231762 157890968768078210 191724747789809255 218169540588403635 232714176627630544 232714176627630544 218169540588403635 191724747789809255 157890968768078210 121801604478231762 87967825456500717 59437719903041025 37539612570341700 22138745874816900 12176310231149295 6236646703759395 2969831763694950 1312251244423350 536830054536825 202802465047245 70539987842520 22512762077400 6566222272575 1742058970275 418094152866 90177170226 17341763505 2944827765 436270780 55525372 5949147 521855 35990 1830 61 1);
                my @pascal_row62 =
                  qw(1 62 1891 37820 557845 6471002 61474519 491796152 3381098545 20286591270 107518933731 508271323092 2160153123141 8308281242850 29078984349975 93052749919920 273342452889765 739632519584070 1849081298960175 4282083008118300 9206478467454345 18412956934908690 34315056105966195 59678358445158600 96977332473382725 147405545359541742 209769429934732479 279692573246309972 349615716557887465 409894288378212890 450883717216034179 465428353255261088 450883717216034179 409894288378212890 349615716557887465 279692573246309972 209769429934732479 147405545359541742 96977332473382725 59678358445158600 34315056105966195 18412956934908690 9206478467454345 4282083008118300 1849081298960175 739632519584070 273342452889765 93052749919920 29078984349975 8308281242850 2160153123141 508271323092 107518933731 20286591270 3381098545 491796152 61474519 6471002 557845 37820 1891 62 1);
                my @pascal_row63 =
                  qw(1 63 1953 39711 595665 7028847 67945521 553270671 3872894697 23667689815 127805525001 615790256823 2668424446233 10468434365991 37387265592825 122131734269895 366395202809685 1012974972473835 2588713818544245 6131164307078475 13488561475572645 27619435402363035 52728013040874885 93993414551124795 156655690918541325 244382877832924467 357174975294274221 489462003181042451 629308289804197437 759510004936100355 860778005594247069 916312070471295267 916312070471295267 860778005594247069 759510004936100355 629308289804197437 489462003181042451 357174975294274221 244382877832924467 156655690918541325 93993414551124795 52728013040874885 27619435402363035 13488561475572645 6131164307078475 2588713818544245 1012974972473835 366395202809685 122131734269895 37387265592825 10468434365991 2668424446233 615790256823 127805525001 23667689815 3872894697 553270671 67945521 7028847 595665 39711 1953 63 1);
                my @pascal_row64 =
                  qw(1 64 2016 41664 635376 7624512 74974368 621216192 4426165368 27540584512 151473214816 743595781824 3284214703056 13136858812224 47855699958816 159518999862720 488526937079580 1379370175283520 3601688791018080 8719878125622720 19619725782651120 41107996877935680 80347448443237920 146721427591999680 250649105469666120 401038568751465792 601557853127198688 846636978475316672 1118770292985239888 1388818294740297792 1620288010530347424 1777090076065542336 1832624140942590534 1777090076065542336 1620288010530347424 1388818294740297792 1118770292985239888 846636978475316672 601557853127198688 401038568751465792 250649105469666120 146721427591999680 80347448443237920 41107996877935680 19619725782651120 8719878125622720 3601688791018080 1379370175283520 488526937079580 159518999862720 47855699958816 13136858812224 3284214703056 743595781824 151473214816 27540584512 4426165368 621216192 74974368 7624512 635376 41664 2016 64 1);
                my @pascal_row65 =
                  qw(1 65 2080 43680 677040 8259888 82598880 696190560 5047381560 31966749880 179013799328 895068996640 4027810484880 16421073515280 60992558771040 207374699821536 648045936942300 1867897112363100 4981058966301600 12321566916640800 28339603908273840 60727722660586800 121455445321173600 227068876035237600 397370533061665800 651687674221131912 1002596421878664480 1448194831602515360 1965407271460556560 2507588587725537680 3009106305270645216 3397378086595889760 3609714217008132870 3609714217008132870 3397378086595889760 3009106305270645216 2507588587725537680 1965407271460556560 1448194831602515360 1002596421878664480 651687674221131912 397370533061665800 227068876035237600 121455445321173600 60727722660586800 28339603908273840 12321566916640800 4981058966301600 1867897112363100 648045936942300 207374699821536 60992558771040 16421073515280 4027810484880 895068996640 179013799328 31966749880 5047381560 696190560 82598880 8259888 677040 43680 2080 65 1);
                my @pascal_row66 =
                  qw(1 66 2145 45760 720720 8936928 90858768 778789440 5743572120 37014131440 210980549208 1074082795968 4922879481520 20448884000160 77413632286320 268367258592576 855420636763836 2515943049305400 6848956078664700 17302625882942400 40661170824914640 89067326568860640 182183167981760400 348524321356411200 624439409096903400 1049058207282797712 1654284096099796392 2450791253481179840 3413602103063071920 4472995859186094240 5516694892996182896 6406484391866534976 7007092303604022630 7219428434016265740 7007092303604022630 6406484391866534976 5516694892996182896 4472995859186094240 3413602103063071920 2450791253481179840 1654284096099796392 1049058207282797712 624439409096903400 348524321356411200 182183167981760400 89067326568860640 40661170824914640 17302625882942400 6848956078664700 2515943049305400 855420636763836 268367258592576 77413632286320 20448884000160 4922879481520 1074082795968 210980549208 37014131440 5743572120 778789440 90858768 8936928 720720 45760 2145 66 1);
                my @pascal_row67 =
                  qw(1 67 2211 47905 766480 9657648 99795696 869648208 6522361560 42757703560 247994680648 1285063345176 5996962277488 25371763481680 97862516286480 345780890878896 1123787895356412 3371363686069236 9364899127970100 24151581961607100 57963796707857040 129728497393775280 271250494550621040 530707489338171600 972963730453314600 1673497616379701112 2703342303382594104 4105075349580976232 5864393356544251760 7886597962249166160 9989690752182277136 11923179284862717872 13413576695470557606 14226520737620288370 14226520737620288370 13413576695470557606 11923179284862717872 9989690752182277136 7886597962249166160 5864393356544251760 4105075349580976232 2703342303382594104 1673497616379701112 972963730453314600 530707489338171600 271250494550621040 129728497393775280 57963796707857040 24151581961607100 9364899127970100 3371363686069236 1123787895356412 345780890878896 97862516286480 25371763481680 5996962277488 1285063345176 247994680648 42757703560 6522361560 869648208 99795696 9657648 766480 47905 2211 67 1);
                my @pascal_row68 =
                  qw(1 68 2278 50116 814385 10424128 109453344 969443904 7392009768 49280065120 290752384208 1533058025824 7282025622664 31368725759168 123234279768160 443643407165376 1469568786235308 4495151581425648 12736262814039336 33516481089577200 82115378669464140 187692294101632320 400978991944396320 801957983888792640 1503671219791486200 2646461346833015712 4376839919762295216 6808417652963570336 9969468706125227992 13750991318793417920 17876288714431443296 21912870037044995008 25336755980333275478 27640097433090845976 28453041475240576740 27640097433090845976 25336755980333275478 21912870037044995008 17876288714431443296 13750991318793417920 9969468706125227992 6808417652963570336 4376839919762295216 2646461346833015712 1503671219791486200 801957983888792640 400978991944396320 187692294101632320 82115378669464140 33516481089577200 12736262814039336 4495151581425648 1469568786235308 443643407165376 123234279768160 31368725759168 7282025622664 1533058025824 290752384208 49280065120 7392009768 969443904 109453344 10424128 814385 50116 2278 68 1);
                my @pascal_row69 =
                  qw(1 69 2346 52394 864501 11238513 119877472 1078897248 8361453672 56672074888 340032449328 1823810410032 8815083648488 38650751381832 154603005527328 566877686933536 1913212193400684 5964720367660956 17231414395464984 46252743903616536 115631859759041340 269807672771096460 588671286046028640 1202936975833188960 2305629203680278840 4150132566624501912 7023301266595310928 11185257572725865552 16777886359088798328 23720460024918645912 31627280033224861216 39789158751476438304 47249626017378270486 52976853413424121454 56093138908331422716 56093138908331422716 52976853413424121454 47249626017378270486 39789158751476438304 31627280033224861216 23720460024918645912 16777886359088798328 11185257572725865552 7023301266595310928 4150132566624501912 2305629203680278840 1202936975833188960 588671286046028640 269807672771096460 115631859759041340 46252743903616536 17231414395464984 5964720367660956 1913212193400684 566877686933536 154603005527328 38650751381832 8815083648488 1823810410032 340032449328 56672074888 8361453672 1078897248 119877472 11238513 864501 52394 2346 69 1);
                my @pascal_row70 =
                  qw(1 70 2415 54740 916895 12103014 131115985 1198774720 9440350920 65033528560 396704524216 2163842859360 10638894058520 47465835030320 193253756909160 721480692460864 2480089880334220 7877932561061640 23196134763125940 63484158299081520 161884603662657876 385439532530137800 858478958817125100 1791608261879217600 3508566179513467800 6455761770304780752 11173433833219812840 18208558839321176480 27963143931814663880 40498346384007444240 55347740058143507128 71416438784701299520 87038784768854708790 100226479430802391940 109069992321755544170 112186277816662845432 109069992321755544170 100226479430802391940 87038784768854708790 71416438784701299520 55347740058143507128 40498346384007444240 27963143931814663880 18208558839321176480 11173433833219812840 6455761770304780752 3508566179513467800 1791608261879217600 858478958817125100 385439532530137800 161884603662657876 63484158299081520 23196134763125940 7877932561061640 2480089880334220 721480692460864 193253756909160 47465835030320 10638894058520 2163842859360 396704524216 65033528560 9440350920 1198774720 131115985 12103014 916895 54740 2415 70 1);
                my @pascal_row71 =
                  qw(1 71 2485 57155 971635 13019909 143218999 1329890705 10639125640 74473879480 461738052776 2560547383576 12802736917880 58104729088840 240719591939480 914734449370024 3201570572795084 10358022441395860 31074067324187580 86680293062207460 225368761961739396 547324136192795676 1243918491347262900 2650087220696342700 5300174441392685400 9964327949818248552 17629195603524593592 29381992672540989320 46171702771135840360 68461490315822108120 95846086442150951368 126764178842844806648 158455223553556008310 187265264199657100730 209296471752557936110 221256270138418389602 221256270138418389602 209296471752557936110 187265264199657100730 158455223553556008310 126764178842844806648 95846086442150951368 68461490315822108120 46171702771135840360 29381992672540989320 17629195603524593592 9964327949818248552 5300174441392685400 2650087220696342700 1243918491347262900 547324136192795676 225368761961739396 86680293062207460 31074067324187580 10358022441395860 3201570572795084 914734449370024 240719591939480 58104729088840 12802736917880 2560547383576 461738052776 74473879480 10639125640 1329890705 143218999 13019909 971635 57155 2485 71 1);
                my @pascal_row72 =
                  qw(1 72 2556 59640 1028790 13991544 156238908 1473109704 11969016345 85113005120 536211932256 3022285436352 15363284301456 70907466006720 298824321028320 1155454041309504 4116305022165108 13559593014190944 41432089765583440 117754360386395040 312049055023946856 772692898154535072 1791242627540058576 3894005712043605600 7950261662089028100 15264502391210933952 27593523553342842144 47011188276065582912 75553695443676829680 114633193086957948480 164307576757973059488 222610265284995758016 285219402396400814958 345720487753213109040 396561735952215036840 430552741890976325712 442512540276836779204 430552741890976325712 396561735952215036840 345720487753213109040 285219402396400814958 222610265284995758016 164307576757973059488 114633193086957948480 75553695443676829680 47011188276065582912 27593523553342842144 15264502391210933952 7950261662089028100 3894005712043605600 1791242627540058576 772692898154535072 312049055023946856 117754360386395040 41432089765583440 13559593014190944 4116305022165108 1155454041309504 298824321028320 70907466006720 15363284301456 3022285436352 536211932256 85113005120 11969016345 1473109704 156238908 13991544 1028790 59640 2556 72 1);
                my @pascal_row73 =
                  qw(1 73 2628 62196 1088430 15020334 170230452 1629348612 13442126049 97082021465 621324937376 3558497368608 18385569737808 86270750308176 369731787035040 1454278362337824 5271759063474612 17675898036356052 54991682779774384 159186450151978480 429803415410341896 1084741953178481928 2563935525694593648 5685248339583664176 11844267374132633700 23214764053299962052 42858025944553776096 74604711829408425056 122564883719742412592 190186888530634778160 278940769844931007968 386917842042968817504 507829667681396572974 630939890149613923998 742282223705428145880 827114477843191362552 873065282167813104916 873065282167813104916 827114477843191362552 742282223705428145880 630939890149613923998 507829667681396572974 386917842042968817504 278940769844931007968 190186888530634778160 122564883719742412592 74604711829408425056 42858025944553776096 23214764053299962052 11844267374132633700 5685248339583664176 2563935525694593648 1084741953178481928 429803415410341896 159186450151978480 54991682779774384 17675898036356052 5271759063474612 1454278362337824 369731787035040 86270750308176 18385569737808 3558497368608 621324937376 97082021465 13442126049 1629348612 170230452 15020334 1088430 62196 2628 73 1);
                my @pascal_row74 =
                  qw(1 74 2701 64824 1150626 16108764 185250786 1799579064 15071474661 110524147514 718406958841 4179822305984 21944067106416 104656320045984 456002537343216 1824010149372864 6726037425812436 22947657099830664 72667580816130436 214178132931752864 588989865562320376 1514545368588823824 3648677478873075576 8249183865278257824 17529515713716297876 35059031427432595752 66072789997853738148 117462737773962201152 197169595549150837648 312751772250377190752 469127658375565786128 665858611887899825472 894747509724365390478 1138769557831010496972 1373222113855042069878 1569396701548619508432 1700179760011004467468 1746130564335626209832 1700179760011004467468 1569396701548619508432 1373222113855042069878 1138769557831010496972 894747509724365390478 665858611887899825472 469127658375565786128 312751772250377190752 197169595549150837648 117462737773962201152 66072789997853738148 35059031427432595752 17529515713716297876 8249183865278257824 3648677478873075576 1514545368588823824 588989865562320376 214178132931752864 72667580816130436 22947657099830664 6726037425812436 1824010149372864 456002537343216 104656320045984 21944067106416 4179822305984 718406958841 110524147514 15071474661 1799579064 185250786 16108764 1150626 64824 2701 74 1);
                my @pascal_row75 =
                  qw(1 75 2775 67525 1215450 17259390 201359550 1984829850 16871053725 125595622175 828931106355 4898229264825 26123889412400 126600387152400 560658857389200 2280012686716080 8550047575185300 29673694525643100 95615237915961100 286845713747883300 803167998494073240 2103535234151144200 5163222847461899400 11897861344151333400 25778699578994555700 52588547141148893628 101131821425286333900 183535527771815939300 314632333323113038800 509921367799528028400 781879430625942976880 1134986270263465611600 1560606121612265215950 2033517067555375887450 2511991671686052566850 2942618815403661578310 3269576461559623975900 3446310324346630677300 3446310324346630677300 3269576461559623975900 2942618815403661578310 2511991671686052566850 2033517067555375887450 1560606121612265215950 1134986270263465611600 781879430625942976880 509921367799528028400 314632333323113038800 183535527771815939300 101131821425286333900 52588547141148893628 25778699578994555700 11897861344151333400 5163222847461899400 2103535234151144200 803167998494073240 286845713747883300 95615237915961100 29673694525643100 8550047575185300 2280012686716080 560658857389200 126600387152400 26123889412400 4898229264825 828931106355 125595622175 16871053725 1984829850 201359550 17259390 1215450 67525 2775 75 1);
                my @pascal_row76 =
                  qw(1 76 2850 70300 1282975 18474840 218618940 2186189400 18855883575 142466675900 954526728530 5727160371180 31022118677225 152724276564800 687259244541600 2840671544105280 10830060261901380 38223742100828400 125288932441604200 382460951663844400 1090013712241956540 2906703232645217440 7266758081613043600 17061084191613232800 37676560923145889100 78367246720143449328 153720368566435227528 284667349197102273200 498167861094928978100 824553701122641067200 1291800798425471005280 1916865700889408588480 2695592391875730827550 3594123189167641103400 4545508739241428454300 5454610487089714145160 6212195276963285554210 6715886785906254653200 6892620648693261354600 6715886785906254653200 6212195276963285554210 5454610487089714145160 4545508739241428454300 3594123189167641103400 2695592391875730827550 1916865700889408588480 1291800798425471005280 824553701122641067200 498167861094928978100 284667349197102273200 153720368566435227528 78367246720143449328 37676560923145889100 17061084191613232800 7266758081613043600 2906703232645217440 1090013712241956540 382460951663844400 125288932441604200 38223742100828400 10830060261901380 2840671544105280 687259244541600 152724276564800 31022118677225 5727160371180 954526728530 142466675900 18855883575 2186189400 218618940 18474840 1282975 70300 2850 76 1);
                my @pascal_row77 =
                  qw(1 77 2926 73150 1353275 19757815 237093780 2404808340 21042072975 161322559475 1096993404430 6681687099710 36749279048405 183746395242025 839983521106400 3527930788646880 13670731806006660 49053802362729780 163512674542432600 507749884105448600 1472474663905800940 3996716944887173980 10173461314258261040 24327842273226276400 54737645114759121900 116043807643289338428 232087615286578676856 438387717763537500728 782835210292031251300 1322721562217570045300 2116354499548112072480 3208666499314879593760 4612458092765139416030 6289715581043371930950 8139631928409069557700 10000119226331142599460 11666805764052999699370 12928082062869540207410 13608507434599516007800 13608507434599516007800 12928082062869540207410 11666805764052999699370 10000119226331142599460 8139631928409069557700 6289715581043371930950 4612458092765139416030 3208666499314879593760 2116354499548112072480 1322721562217570045300 782835210292031251300 438387717763537500728 232087615286578676856 116043807643289338428 54737645114759121900 24327842273226276400 10173461314258261040 3996716944887173980 1472474663905800940 507749884105448600 163512674542432600 49053802362729780 13670731806006660 3527930788646880 839983521106400 183746395242025 36749279048405 6681687099710 1096993404430 161322559475 21042072975 2404808340 237093780 19757815 1353275 73150 2926 77 1);
                my @pascal_row78 =
                  qw(1 78 3003 76076 1426425 21111090 256851595 2641902120 23446881315 182364632450 1258315963905 7778680504140 43430966148115 220495674290430 1023729916348425 4367914309753280 17198662594653540 62724534168736440 212566476905162380 671262558647881200 1980224548011249540 5469191608792974920 14170178259145435020 34501303587484537440 79065487387985398300 170781452758048460328 348131422929868015284 670475333050116177584 1221222928055568752028 2105556772509601296600 3439076061765682117780 5325020998862991666240 7821124592080019009790 10902173673808511346980 14429347509452441488650 18139751154740212157160 21666924990384142298830 24594887826922539906780 26536589497469056215210 27217014869199032015600 26536589497469056215210 24594887826922539906780 21666924990384142298830 18139751154740212157160 14429347509452441488650 10902173673808511346980 7821124592080019009790 5325020998862991666240 3439076061765682117780 2105556772509601296600 1221222928055568752028 670475333050116177584 348131422929868015284 170781452758048460328 79065487387985398300 34501303587484537440 14170178259145435020 5469191608792974920 1980224548011249540 671262558647881200 212566476905162380 62724534168736440 17198662594653540 4367914309753280 1023729916348425 220495674290430 43430966148115 7778680504140 1258315963905 182364632450 23446881315 2641902120 256851595 21111090 1426425 76076 3003 78 1);
                my @pascal_row79 =
                  qw(1 79 3081 79079 1502501 22537515 277962685 2898753715 26088783435 205811513765 1440680596355 9036996468045 51209646652255 263926640438545 1244225590638855 5391644226101705 21566576904406820 79923196763389980 275291011073898820 883829035553043580 2651487106659130740 7449416156804224460 19639369867938409940 48671481846629972460 113566790975469935740 249846940146033858628 518912875687916475612 1018606755979984192868 1891698261105684929612 3326779700565170048628 5544632834275283414380 8764097060628673784020 13146145590943010676030 18723298265888530356770 25331521183260952835630 32569098664192653645810 39806676145124354455990 46261812817306682205610 51131477324391596121990 53753604366668088230810 53753604366668088230810 51131477324391596121990 46261812817306682205610 39806676145124354455990 32569098664192653645810 25331521183260952835630 18723298265888530356770 13146145590943010676030 8764097060628673784020 5544632834275283414380 3326779700565170048628 1891698261105684929612 1018606755979984192868 518912875687916475612 249846940146033858628 113566790975469935740 48671481846629972460 19639369867938409940 7449416156804224460 2651487106659130740 883829035553043580 275291011073898820 79923196763389980 21566576904406820 5391644226101705 1244225590638855 263926640438545 51209646652255 9036996468045 1440680596355 205811513765 26088783435 2898753715 277962685 22537515 1502501 79079 3081 79 1);
                my @pascal_row80 =
                  qw(1 80 3160 82160 1581580 24040016 300500200 3176716400 28987537150 231900297200 1646492110120 10477677064400 60246643120300 315136287090800 1508152231077400 6635869816740560 26958221130508525 101489773667796800 355214207837288800 1159120046626942400 3535316142212174320 10100903263463355200 27088786024742634400 68310851714568382400 162238272822099908200 363413731121503794368 768759815833950334240 1537519631667900668480 2910305017085669122480 5218477961670854978240 8871412534840453463008 14308729894903957198400 21910242651571684460050 31869443856831541032800 44054819449149483192400 57900619847453606481440 72375774809317008101800 86068488962431036661600 97393290141698278327600 104885081691059684352800 107507208733336176461620 104885081691059684352800 97393290141698278327600 86068488962431036661600 72375774809317008101800 57900619847453606481440 44054819449149483192400 31869443856831541032800 21910242651571684460050 14308729894903957198400 8871412534840453463008 5218477961670854978240 2910305017085669122480 1537519631667900668480 768759815833950334240 363413731121503794368 162238272822099908200 68310851714568382400 27088786024742634400 10100903263463355200 3535316142212174320 1159120046626942400 355214207837288800 101489773667796800 26958221130508525 6635869816740560 1508152231077400 315136287090800 60246643120300 10477677064400 1646492110120 231900297200 28987537150 3176716400 300500200 24040016 1581580 82160 3160 80 1);
                my @pascal_row81 =
                  qw(1 81 3240 85320 1663740 25621596 324540216 3477216600 32164253550 260887834350 1878392407320 12124169174520 70724320184700 375382930211100 1823288518168200 8144022047817960 33594090947249085 128447994798305325 456703981505085600 1514334254464231200 4694436188839116720 13636219405675529520 37189689288205989600 95399637739311016800 230549124536668290600 525652003943603702568 1132173546955454128608 2306279447501851002720 4447824648753569790960 8128782978756524100720 14089890496511308441248 23180142429744410661408 36218972546475641658450 53779686508403225492850 75924263305981024225200 101955439296603089673840 130276394656770614583240 158444263771748044763400 183461779104129314989200 202278371832757962680400 212392290424395860814420 212392290424395860814420 202278371832757962680400 183461779104129314989200 158444263771748044763400 130276394656770614583240 101955439296603089673840 75924263305981024225200 53779686508403225492850 36218972546475641658450 23180142429744410661408 14089890496511308441248 8128782978756524100720 4447824648753569790960 2306279447501851002720 1132173546955454128608 525652003943603702568 230549124536668290600 95399637739311016800 37189689288205989600 13636219405675529520 4694436188839116720 1514334254464231200 456703981505085600 128447994798305325 33594090947249085 8144022047817960 1823288518168200 375382930211100 70724320184700 12124169174520 1878392407320 260887834350 32164253550 3477216600 324540216 25621596 1663740 85320 3240 81 1);
                my @pascal_row82 =
                  qw(1 82 3321 88560 1749060 27285336 350161812 3801756816 35641470150 293052087900 2139280241670 14002561581840 82848489359220 446107250395800 2198671448379300 9967310565986160 41738112995067045 162042085745554410 585151976303390925 1971038235969316800 6208770443303347920 18330655594514646240 50825908693881519120 132589327027517006400 325948762275979307400 756201128480271993168 1657825550899057831176 3438452994457305131328 6754104096255420793680 12576607627510093891680 22218673475267832541968 37270032926255719102656 59399114976220052319858 89998659054878867151300 129703949814384249718050 177879702602584113899040 232231833953373704257080 288720658428518659346640 341906042875877359752600 385740150936887277669600 414670662257153823494820 424784580848791721628840 414670662257153823494820 385740150936887277669600 341906042875877359752600 288720658428518659346640 232231833953373704257080 177879702602584113899040 129703949814384249718050 89998659054878867151300 59399114976220052319858 37270032926255719102656 22218673475267832541968 12576607627510093891680 6754104096255420793680 3438452994457305131328 1657825550899057831176 756201128480271993168 325948762275979307400 132589327027517006400 50825908693881519120 18330655594514646240 6208770443303347920 1971038235969316800 585151976303390925 162042085745554410 41738112995067045 9967310565986160 2198671448379300 446107250395800 82848489359220 14002561581840 2139280241670 293052087900 35641470150 3801756816 350161812 27285336 1749060 88560 3321 82 1);
                my @pascal_row83 =
                  qw(1 83 3403 91881 1837620 29034396 377447148 4151918628 39443226966 328693558050 2432332329570 16141841823510 96851050941060 528955739755020 2644778698775100 12165982014365460 51705423561053205 203780198740621455 747194062048945335 2556190212272707725 8179808679272664720 24539426037817994160 69156564288396165360 183415235721398525520 458538089303496313800 1082149890756251300568 2414026679379329824344 5096278545356362962504 10192557090712725925008 19330711723765514685360 34795281102777926433648 59488706401523551644624 96669147902475771422514 149397774031098919471158 219702608869263116869350 307583652416968363617090 410111536555957818156120 520952492381892363603720 630626701304396019099240 727646193812764637422200 800410813194041101164420 839455243105945545123660 839455243105945545123660 800410813194041101164420 727646193812764637422200 630626701304396019099240 520952492381892363603720 410111536555957818156120 307583652416968363617090 219702608869263116869350 149397774031098919471158 96669147902475771422514 59488706401523551644624 34795281102777926433648 19330711723765514685360 10192557090712725925008 5096278545356362962504 2414026679379329824344 1082149890756251300568 458538089303496313800 183415235721398525520 69156564288396165360 24539426037817994160 8179808679272664720 2556190212272707725 747194062048945335 203780198740621455 51705423561053205 12165982014365460 2644778698775100 528955739755020 96851050941060 16141841823510 2432332329570 328693558050 39443226966 4151918628 377447148 29034396 1837620 91881 3403 83 1);
                my @pascal_row84 =
                  qw(1 84 3486 95284 1929501 30872016 406481544 4529365776 43595145594 368136785016 2761025887620 18574174153080 112992892764570 625806790696080 3173734438530120 14810760713140560 63871405575418665 255485622301674660 950974260789566790 3303384274321653060 10735998891545372445 32719234717090658880 93695990326214159520 252571800009794690880 641953325024894839320 1540687980059747614368 3496176570135581124912 7510305224735692786848 15288835636069088887512 29523268814478240610368 54125992826543441119008 94283987504301478078272 156157854303999323067138 246066921933574690893672 369100382900362036340508 527286261286231480486440 717695188972926181773210 931064028937850181759840 1151579193686288382702960 1358272895117160656521440 1528057007006805738586620 1639866056299986646288080 1678910486211891090247320 1639866056299986646288080 1528057007006805738586620 1358272895117160656521440 1151579193686288382702960 931064028937850181759840 717695188972926181773210 527286261286231480486440 369100382900362036340508 246066921933574690893672 156157854303999323067138 94283987504301478078272 54125992826543441119008 29523268814478240610368 15288835636069088887512 7510305224735692786848 3496176570135581124912 1540687980059747614368 641953325024894839320 252571800009794690880 93695990326214159520 32719234717090658880 10735998891545372445 3303384274321653060 950974260789566790 255485622301674660 63871405575418665 14810760713140560 3173734438530120 625806790696080 112992892764570 18574174153080 2761025887620 368136785016 43595145594 4529365776 406481544 30872016 1929501 95284 3486 84 1);
                my @pascal_row85 =
                  qw(1 85 3570 98770 2024785 32801517 437353560 4935847320 48124511370 411731930610 3129162672636 21335200040700 131567066917650 738799683460650 3799541229226200 17984495151670680 78682166288559225 319357027877093325 1206459883091241450 4254358535111219850 14039383165867025505 43455233608636031325 126415225043304818400 346267790336008850400 894525125034689530200 2182641305084642453688 5036864550195328739280 11006481794871273911760 22799140860804781674360 44812104450547329497880 83649261641021681729376 148409980330844919197280 250441841808300801145410 402224776237574013960810 615167304833936727234180 896386644186593516826948 1244981450259157662259650 1648759217910776363533050 2082643222624138564462800 2509852088803449039224400 2886329902123966395108060 3167923063306792384874700 3318776542511877736535400 3318776542511877736535400 3167923063306792384874700 2886329902123966395108060 2509852088803449039224400 2082643222624138564462800 1648759217910776363533050 1244981450259157662259650 896386644186593516826948 615167304833936727234180 402224776237574013960810 250441841808300801145410 148409980330844919197280 83649261641021681729376 44812104450547329497880 22799140860804781674360 11006481794871273911760 5036864550195328739280 2182641305084642453688 894525125034689530200 346267790336008850400 126415225043304818400 43455233608636031325 14039383165867025505 4254358535111219850 1206459883091241450 319357027877093325 78682166288559225 17984495151670680 3799541229226200 738799683460650 131567066917650 21335200040700 3129162672636 411731930610 48124511370 4935847320 437353560 32801517 2024785 98770 3570 85 1);
                my @pascal_row86 =
                  qw(1 86 3655 102340 2123555 34826302 470155077 5373200880 53060358690 459856441980 3540894603246 24464362713336 152902266958350 870366750378300 4538340912686850 21784036380896880 96666661440229905 398039194165652550 1525816910968334775 5460818418202461300 18293741700978245355 57494616774503056830 169870458651940849725 472683015379313668800 1240792915370698380600 3077166430119331983888 7219505855279971192968 16043346345066602651040 33805622655676055586120 67611245311352111172240 128461366091569011227256 232059241971866600926656 398851822139145720342690 652666618045874815106220 1017392081071510741194990 1511553949020530244061128 2141368094445751179086598 2893740668169934025792700 3731402440534914927995850 4592495311427587603687200 5396181990927415434332460 6054252965430758779982760 6486699605818670121410100 6637553085023755473070800 6486699605818670121410100 6054252965430758779982760 5396181990927415434332460 4592495311427587603687200 3731402440534914927995850 2893740668169934025792700 2141368094445751179086598 1511553949020530244061128 1017392081071510741194990 652666618045874815106220 398851822139145720342690 232059241971866600926656 128461366091569011227256 67611245311352111172240 33805622655676055586120 16043346345066602651040 7219505855279971192968 3077166430119331983888 1240792915370698380600 472683015379313668800 169870458651940849725 57494616774503056830 18293741700978245355 5460818418202461300 1525816910968334775 398039194165652550 96666661440229905 21784036380896880 4538340912686850 870366750378300 152902266958350 24464362713336 3540894603246 459856441980 53060358690 5373200880 470155077 34826302 2123555 102340 3655 86 1);
                my @pascal_row87 =
                  qw(1 87 3741 105995 2225895 36949857 504981379 5843355957 58433559570 512916800670 4000751045226 28005257316582 177366629671686 1023269017336650 5408707663065150 26322377293583730 118450697821126785 494705855605882455 1923856105133987325 6986635329170796075 23754560119180706655 75788358475481302185 227365075426443906555 642553474031254518525 1713475930750012049400 4317959345490030364488 10296672285399303176856 23262852200346573844008 49848969000742658237160 101416867967028166758360 196072611402921122399496 360520608063435612153912 630911064111012321269346 1051518440185020535448910 1670058699117385556301210 2528946030092040985256118 3652922043466281423147726 5035108762615685204879298 6625143108704848953788550 8323897751962502531683050 9988677302355003038019660 11450434956358174214315220 12540952571249428901392860 13124252690842425594480900 13124252690842425594480900 12540952571249428901392860 11450434956358174214315220 9988677302355003038019660 8323897751962502531683050 6625143108704848953788550 5035108762615685204879298 3652922043466281423147726 2528946030092040985256118 1670058699117385556301210 1051518440185020535448910 630911064111012321269346 360520608063435612153912 196072611402921122399496 101416867967028166758360 49848969000742658237160 23262852200346573844008 10296672285399303176856 4317959345490030364488 1713475930750012049400 642553474031254518525 227365075426443906555 75788358475481302185 23754560119180706655 6986635329170796075 1923856105133987325 494705855605882455 118450697821126785 26322377293583730 5408707663065150 1023269017336650 177366629671686 28005257316582 4000751045226 512916800670 58433559570 5843355957 504981379 36949857 2225895 105995 3741 87 1);
                my @pascal_row88 =
                  qw(1 88 3828 109736 2331890 39175752 541931236 6348337336 64276915527 571350360240 4513667845896 32006008361808 205371886988268 1200635647008336 6431976680401800 31731084956648880 144773075114710515 613156553427009240 2418561960739869780 8910491434304783400 30741195448351502730 99542918594662008840 303153433901925208740 869918549457698425080 2356029404781266567925 6031435276240042413888 14614631630889333541344 33559524485745877020864 73111821201089232081168 151265836967770824995520 297489479369949289157856 556593219466356734553408 991431672174447933423258 1682429504296032856718256 2721577139302406091750120 4199004729209426541557328 6181868073558322408403844 8688030806081966628027024 11660251871320534158667848 14949040860667351485471600 18312575054317505569702710 21439112258713177252334880 23991387527607603115708080 25665205262091854495873760 26248505381684851188961800 25665205262091854495873760 23991387527607603115708080 21439112258713177252334880 18312575054317505569702710 14949040860667351485471600 11660251871320534158667848 8688030806081966628027024 6181868073558322408403844 4199004729209426541557328 2721577139302406091750120 1682429504296032856718256 991431672174447933423258 556593219466356734553408 297489479369949289157856 151265836967770824995520 73111821201089232081168 33559524485745877020864 14614631630889333541344 6031435276240042413888 2356029404781266567925 869918549457698425080 303153433901925208740 99542918594662008840 30741195448351502730 8910491434304783400 2418561960739869780 613156553427009240 144773075114710515 31731084956648880 6431976680401800 1200635647008336 205371886988268 32006008361808 4513667845896 571350360240 64276915527 6348337336 541931236 39175752 2331890 109736 3828 88 1);
                my @pascal_row89 =
                  qw(1 89 3916 113564 2441626 41507642 581106988 6890268572 70625252863 635627275767 5085018206136 36519676207704 237377895350076 1406007533996604 7632612327410136 38163061637050680 176504160071359395 757929628541719755 3031718514166879020 11329053395044653180 39651686882656286130 130284114043013511570 402696352496587217580 1173071983359623633820 3225947954238964993005 8387464681021308981813 20646066907129375955232 48174156116635210562208 106671345686835109102032 224377658168860057076688 448755316337720114153376 854082698836306023711264 1548024891640804667976666 2673861176470480790141514 4404006643598438948468376 6920581868511832633307448 10380872802767748949961172 14869898879640289036430868 20348282677402500786694872 26609292731987885644139448 33261615914984857055174310 39751687313030682822037590 45430499786320780368042960 49656592789699457611581840 51913710643776705684835560 51913710643776705684835560 49656592789699457611581840 45430499786320780368042960 39751687313030682822037590 33261615914984857055174310 26609292731987885644139448 20348282677402500786694872 14869898879640289036430868 10380872802767748949961172 6920581868511832633307448 4404006643598438948468376 2673861176470480790141514 1548024891640804667976666 854082698836306023711264 448755316337720114153376 224377658168860057076688 106671345686835109102032 48174156116635210562208 20646066907129375955232 8387464681021308981813 3225947954238964993005 1173071983359623633820 402696352496587217580 130284114043013511570 39651686882656286130 11329053395044653180 3031718514166879020 757929628541719755 176504160071359395 38163061637050680 7632612327410136 1406007533996604 237377895350076 36519676207704 5085018206136 635627275767 70625252863 6890268572 581106988 41507642 2441626 113564 3916 89 1);
                my @pascal_row90 =
                  qw(1 90 4005 117480 2555190 43949268 622614630 7471375560 77515521435 706252528630 5720645481903 41604694413840 273897571557780 1643385429346680 9038619861406740 45795673964460816 214667221708410075 934433788613079150 3789648142708598775 14360771909211532200 50980740277700939310 169935800925669797700 532980466539600729150 1575768335856210851400 4399019937598588626825 11613412635260273974818 29033531588150684937045 68820223023764586517440 154845501803470319664240 331049003855695166178720 673132974506580171230064 1302838015174026137864640 2402107590477110691687930 4221886068111285458118180 7077867820068919738609890 11324588512110271581775824 17301454671279581583268620 25250771682408037986392040 35218181557042789823125740 46957575409390386430834320 59870908646972742699313758 73013303228015539877211900 85182187099351463190080550 95087092576020237979624800 101570303433476163296417400 103827421287553411369671120 101570303433476163296417400 95087092576020237979624800 85182187099351463190080550 73013303228015539877211900 59870908646972742699313758 46957575409390386430834320 35218181557042789823125740 25250771682408037986392040 17301454671279581583268620 11324588512110271581775824 7077867820068919738609890 4221886068111285458118180 2402107590477110691687930 1302838015174026137864640 673132974506580171230064 331049003855695166178720 154845501803470319664240 68820223023764586517440 29033531588150684937045 11613412635260273974818 4399019937598588626825 1575768335856210851400 532980466539600729150 169935800925669797700 50980740277700939310 14360771909211532200 3789648142708598775 934433788613079150 214667221708410075 45795673964460816 9038619861406740 1643385429346680 273897571557780 41604694413840 5720645481903 706252528630 77515521435 7471375560 622614630 43949268 2555190 117480 4005 90 1);
                my @pascal_row91 =
                  qw(1 91 4095 121485 2672670 46504458 666563898 8093990190 84986896995 783768050065 6426898010533 47325339895743 315502265971620 1917283000904460 10682005290753420 54834293825867556 260462895672870891 1149101010321489225 4724081931321677925 18150420051920130975 65341512186912471510 220916541203370737010 702916267465270526850 2108748802395811580550 5974788273454799478225 16012432572858862601643 40646944223410958911863 97853754611915271454485 223665724827234906181680 485894505659165485842960 1004181978362275337408784 1975970989680606309094704 3704945605651136829552570 6623993658588396149806110 11299753888180205196728070 18402456332179191320385714 28626043183389853165044444 42552226353687619569660660 60468953239450827809517780 82175756966433176253960060 106828484056363129130148078 132884211874988282576525658 158195490327367003067292450 180269279675371701169705350 196657396009496401276042200 205397724721029574666088520 205397724721029574666088520 196657396009496401276042200 180269279675371701169705350 158195490327367003067292450 132884211874988282576525658 106828484056363129130148078 82175756966433176253960060 60468953239450827809517780 42552226353687619569660660 28626043183389853165044444 18402456332179191320385714 11299753888180205196728070 6623993658588396149806110 3704945605651136829552570 1975970989680606309094704 1004181978362275337408784 485894505659165485842960 223665724827234906181680 97853754611915271454485 40646944223410958911863 16012432572858862601643 5974788273454799478225 2108748802395811580550 702916267465270526850 220916541203370737010 65341512186912471510 18150420051920130975 4724081931321677925 1149101010321489225 260462895672870891 54834293825867556 10682005290753420 1917283000904460 315502265971620 47325339895743 6426898010533 783768050065 84986896995 8093990190 666563898 46504458 2672670 121485 4095 91 1);
                my @pascal_row92 =
                  qw(1 92 4186 125580 2794155 49177128 713068356 8760554088 93080887185 868754947060 7210666060598 53752237906276 362827605867363 2232785266876080 12599288291657880 65516299116620976 315297189498738447 1409563905994360116 5873182941643167150 22874501983241808900 83491932238832602485 286258053390283208520 923832808668641263860 2811665069861082107400 8083537075850611058775 21987220846313662079868 56659376796269821513506 138500698835326230366348 321519479439150177636165 709560230486400392024640 1490076484021440823251744 2980152968042881646503488 5680916595331743138647274 10328939264239532979358680 17923747546768601346534180 29702210220359396517113784 47028499515569044485430158 71178269537077472734705104 103021179593138447379178440 142644710205884004063477840 189004241022796305384108138 239712695931351411706673736 291079702202355285643818108 338464770002738704236997800 376926675684868102445747550 402055120730525975942130720 410795449442059149332177040 402055120730525975942130720 376926675684868102445747550 338464770002738704236997800 291079702202355285643818108 239712695931351411706673736 189004241022796305384108138 142644710205884004063477840 103021179593138447379178440 71178269537077472734705104 47028499515569044485430158 29702210220359396517113784 17923747546768601346534180 10328939264239532979358680 5680916595331743138647274 2980152968042881646503488 1490076484021440823251744 709560230486400392024640 321519479439150177636165 138500698835326230366348 56659376796269821513506 21987220846313662079868 8083537075850611058775 2811665069861082107400 923832808668641263860 286258053390283208520 83491932238832602485 22874501983241808900 5873182941643167150 1409563905994360116 315297189498738447 65516299116620976 12599288291657880 2232785266876080 362827605867363 53752237906276 7210666060598 868754947060 93080887185 8760554088 713068356 49177128 2794155 125580 4186 92 1);
                my @pascal_row93 =
                  qw(1 93 4278 129766 2919735 51971283 762245484 9473622444 101841441273 961835834245 8079421007658 60962903966874 416579843773639 2595612872743443 14832073558533960 78115587408278856 380813488615359423 1724861095493098563 7282746847637527266 28747684924884976050 106366434222074411385 369749985629115811005 1210090862058924472380 3735497878529723371260 10895202145711693166175 30070757922164273138643 78646597642583483593374 195160075631596051879854 460020178274476408002513 1031079709925550569660805 2199636714507841215276384 4470229452064322469755232 8661069563374624785150762 16009855859571276118005954 28252686811008134325892860 47625957767127997863647964 76730709735928441002543942 118206769052646517220135262 174199449130215920113883544 245665889799022451442656280 331648951228680309447585978 428716936954147717090781874 530792398133706697350491844 629544472205093989880815908 715391445687606806682745350 778981796415394078387878270 812850570172585125274307760 812850570172585125274307760 778981796415394078387878270 715391445687606806682745350 629544472205093989880815908 530792398133706697350491844 428716936954147717090781874 331648951228680309447585978 245665889799022451442656280 174199449130215920113883544 118206769052646517220135262 76730709735928441002543942 47625957767127997863647964 28252686811008134325892860 16009855859571276118005954 8661069563374624785150762 4470229452064322469755232 2199636714507841215276384 1031079709925550569660805 460020178274476408002513 195160075631596051879854 78646597642583483593374 30070757922164273138643 10895202145711693166175 3735497878529723371260 1210090862058924472380 369749985629115811005 106366434222074411385 28747684924884976050 7282746847637527266 1724861095493098563 380813488615359423 78115587408278856 14832073558533960 2595612872743443 416579843773639 60962903966874 8079421007658 961835834245 101841441273 9473622444 762245484 51971283 2919735 129766 4278 93 1);
                my @pascal_row94 =
                  qw(1 94 4371 134044 3049501 54891018 814216767 10235867928 111315063717 1063677275518 9041256841903 69042324974532 477542747740513 3012192716517082 17427686431277403 92947660966812816 458929076023638279 2105674584108457986 9007607943130625829 36030431772522503316 135114119146959387435 476116419851190222390 1579840847688040283385 4945588740588647843640 14630700024241416537435 40965960067875966304818 108717355564747756732017 273806673274179535473228 655180253906072459882367 1491099888200026977663318 3230716424433391784937189 6669866166572163685031616 13131299015438947254905994 24670925422945900903156716 44262542670579410443898814 75878644578136132189540824 124356667503056438866191906 194937478788574958222679204 292406218182862437334018806 419865338929238371556539824 577314841027702760890242258 760365888182828026538367852 959509335087854414441273718 1160336870338800687231307752 1344935917892700796563561258 1494373242103000885070623620 1591832366587979203662186030 1625701140345170250548615520 1591832366587979203662186030 1494373242103000885070623620 1344935917892700796563561258 1160336870338800687231307752 959509335087854414441273718 760365888182828026538367852 577314841027702760890242258 419865338929238371556539824 292406218182862437334018806 194937478788574958222679204 124356667503056438866191906 75878644578136132189540824 44262542670579410443898814 24670925422945900903156716 13131299015438947254905994 6669866166572163685031616 3230716424433391784937189 1491099888200026977663318 655180253906072459882367 273806673274179535473228 108717355564747756732017 40965960067875966304818 14630700024241416537435 4945588740588647843640 1579840847688040283385 476116419851190222390 135114119146959387435 36030431772522503316 9007607943130625829 2105674584108457986 458929076023638279 92947660966812816 17427686431277403 3012192716517082 477542747740513 69042324974532 9041256841903 1063677275518 111315063717 10235867928 814216767 54891018 3049501 134044 4371 94 1);
                my @pascal_row95 =
                  qw(1 95 4465 138415 3183545 57940519 869107785 11050084695 121550931645 1174992339235 10104934117421 78083581816435 546585072715045 3489735464257595 20439879147794485 110375347398090219 551876736990451095 2564603660132096265 11113282527239083815 45038039715653129145 171144550919481890751 611230538998149609825 2055957267539230505775 6525429588276688127025 19576288764830064381075 55596660092117382842253 149683315632623723036835 382524028838927292205245 928986927180251995355595 2146280142106099437545685 4721816312633418762600507 9900582591005555469968805 19801165182011110939937610 37802224438384848158062710 68933468093525311347055530 120141187248715542633439638 200235312081192571055732730 319294146291631397088871110 487343696971437395556698010 712271557112100808890558630 997180179956941132446782082 1337680729210530787428610110 1719875223270682440979641570 2119846205426655101672581470 2505272788231501483794869010 2839309159995701681634184878 3086205608690980088732809650 3217533506933149454210801550 3217533506933149454210801550 3086205608690980088732809650 2839309159995701681634184878 2505272788231501483794869010 2119846205426655101672581470 1719875223270682440979641570 1337680729210530787428610110 997180179956941132446782082 712271557112100808890558630 487343696971437395556698010 319294146291631397088871110 200235312081192571055732730 120141187248715542633439638 68933468093525311347055530 37802224438384848158062710 19801165182011110939937610 9900582591005555469968805 4721816312633418762600507 2146280142106099437545685 928986927180251995355595 382524028838927292205245 149683315632623723036835 55596660092117382842253 19576288764830064381075 6525429588276688127025 2055957267539230505775 611230538998149609825 171144550919481890751 45038039715653129145 11113282527239083815 2564603660132096265 551876736990451095 110375347398090219 20439879147794485 3489735464257595 546585072715045 78083581816435 10104934117421 1174992339235 121550931645 11050084695 869107785 57940519 3183545 138415 4465 95 1);
                my @pascal_row96 =
                  qw(1 96 4560 142880 3321960 61124064 927048304 11919192480 132601016340 1296543270880 11279926456656 88188515933856 624668654531480 4036320536972640 23929614612052080 130815226545884704 662252084388541314 3116480397122547360 13677886187371180080 56151322242892212960 216182590635135019896 782375089917631500576 2667187806537380115600 8581386855815918632800 26101718353106752508100 75172948856947447223328 205279975724741105879088 532207344471551015242080 1311510956019179287560840 3075267069286351432901280 6868096454739518200146192 14622398903638974232569312 29701747773016666409906415 57603389620395959098000320 106735692531910159505118240 189074655342240853980495168 320376499329908113689172368 519529458372823968144603840 806637843263068792645569120 1199615254083538204447256640 1709451737069041941337340712 2334860909167471919875392192 3057555952481213228408251680 3839721428697337542652223040 4625118993658156585467450480 5344581948227203165429053888 5925514768686681770366994528 6303739115624129542943611200 6435067013866298908421603100 6303739115624129542943611200 5925514768686681770366994528 5344581948227203165429053888 4625118993658156585467450480 3839721428697337542652223040 3057555952481213228408251680 2334860909167471919875392192 1709451737069041941337340712 1199615254083538204447256640 806637843263068792645569120 519529458372823968144603840 320376499329908113689172368 189074655342240853980495168 106735692531910159505118240 57603389620395959098000320 29701747773016666409906415 14622398903638974232569312 6868096454739518200146192 3075267069286351432901280 1311510956019179287560840 532207344471551015242080 205279975724741105879088 75172948856947447223328 26101718353106752508100 8581386855815918632800 2667187806537380115600 782375089917631500576 216182590635135019896 56151322242892212960 13677886187371180080 3116480397122547360 662252084388541314 130815226545884704 23929614612052080 4036320536972640 624668654531480 88188515933856 11279926456656 1296543270880 132601016340 11919192480 927048304 61124064 3321960 142880 4560 96 1);
                my @pascal_row97 =
                  qw(1 97 4656 147440 3464840 64446024 988172368 12846240784 144520208820 1429144287220 12576469727536 99468442390512 712857170465336 4660989191504120 27965935149024720 154744841157936784 793067310934426018 3778732481511088674 16794366584493727440 69829208430263393040 272333912878027232856 998557680552766520472 3449562896455011616176 11248574662353298748400 34683105208922671140900 101274667210054199731428 280452924581688553102416 737487320196292121121168 1843718300490730302802920 4386778025305530720462120 9943363524025869633047472 21490495358378492432715504 44324146676655640642475727 87305137393412625507906735 164339082152306118603118560 295810347874151013485613408 509451154672148967669667536 839905957702732081833776208 1326167301635892760790172960 2006253097346606997092825760 2909066991152580145784597352 4044312646236513861212732904 5392416861648685148283643872 6897277381178550771060474720 8464840422355494128119673520 9969700941885359750896504368 11270096716913884935796048416 12229253884310811313310605728 12738806129490428451365214300 12738806129490428451365214300 12229253884310811313310605728 11270096716913884935796048416 9969700941885359750896504368 8464840422355494128119673520 6897277381178550771060474720 5392416861648685148283643872 4044312646236513861212732904 2909066991152580145784597352 2006253097346606997092825760 1326167301635892760790172960 839905957702732081833776208 509451154672148967669667536 295810347874151013485613408 164339082152306118603118560 87305137393412625507906735 44324146676655640642475727 21490495358378492432715504 9943363524025869633047472 4386778025305530720462120 1843718300490730302802920 737487320196292121121168 280452924581688553102416 101274667210054199731428 34683105208922671140900 11248574662353298748400 3449562896455011616176 998557680552766520472 272333912878027232856 69829208430263393040 16794366584493727440 3778732481511088674 793067310934426018 154744841157936784 27965935149024720 4660989191504120 712857170465336 99468442390512 12576469727536 1429144287220 144520208820 12846240784 988172368 64446024 3464840 147440 4656 97 1);
                my @pascal_row98 =
                  qw(1 98 4753 152096 3612280 67910864 1052618392 13834413152 157366449604 1573664496040 14005614014756 112044912118048 812325612855848 5373846361969456 32626924340528840 182710776306961504 947812152092362802 4571799792445514692 20573099066004816114 86623575014757120480 342163121308290625896 1270891593430793753328 4448120577007778136648 14698137558808310364576 45931679871275969889300 135957772418976870872328 381727591791742752833844 1017940244777980674223584 2581205620687022423924088 6230496325796261023265040 14330141549331400353509592 31433858882404362065762976 65814642035034133075191231 131629284070068266150382462 251644219545718744111025295 460149430026457132088731968 805261502546299981155280944 1349357112374881049503443744 2166073259338624842623949168 3332420398982499757882998720 4915320088499187142877423112 6953379637389094006997330256 9436729507885199009496376776 12289694242827235919344118592 15362117803534044899180148240 18434541364240853879016177888 21239797658799244686692552784 23499350601224696249106654144 24968060013801239764675820028 25477612258980856902730428600 24968060013801239764675820028 23499350601224696249106654144 21239797658799244686692552784 18434541364240853879016177888 15362117803534044899180148240 12289694242827235919344118592 9436729507885199009496376776 6953379637389094006997330256 4915320088499187142877423112 3332420398982499757882998720 2166073259338624842623949168 1349357112374881049503443744 805261502546299981155280944 460149430026457132088731968 251644219545718744111025295 131629284070068266150382462 65814642035034133075191231 31433858882404362065762976 14330141549331400353509592 6230496325796261023265040 2581205620687022423924088 1017940244777980674223584 381727591791742752833844 135957772418976870872328 45931679871275969889300 14698137558808310364576 4448120577007778136648 1270891593430793753328 342163121308290625896 86623575014757120480 20573099066004816114 4571799792445514692 947812152092362802 182710776306961504 32626924340528840 5373846361969456 812325612855848 112044912118048 14005614014756 1573664496040 157366449604 13834413152 1052618392 67910864 3612280 152096 4753 98 1);
                my @pascal_row99 =
                  qw(1 99 4851 156849 3764376 71523144 1120529256 14887031544 171200862756 1731030945644 15579278510796 126050526132804 924370524973896 6186171974825304 38000770702498296 215337700647490344 1130522928399324306 5519611944537877494 25144898858450330806 107196674080761936594 428786696323047746376 1613054714739084379224 5719012170438571889976 19146258135816088501224 60629817430084280253876 181889452290252840761628 517685364210719623706172 1399667836569723427057428 3599145865465003098147672 8811701946483283447189128 20560637875127661376774632 45764000431735762419272568 97248500917438495140954207 197443926105102399225573693 383273503615787010261407757 711793649572175876199757263 1265410932572757113244012912 2154618614921181030658724688 3515430371713505892127392912 5498493658321124600506947888 8247740487481686900760421832 11868699725888281149874753368 16390109145274293016493707032 21726423750712434928840495368 27651812046361280818524266832 33796659167774898778196326128 39674339023040098565708730672 44739148260023940935799206928 48467410615025936013782474172 50445672272782096667406248628 50445672272782096667406248628 48467410615025936013782474172 44739148260023940935799206928 39674339023040098565708730672 33796659167774898778196326128 27651812046361280818524266832 21726423750712434928840495368 16390109145274293016493707032 11868699725888281149874753368 8247740487481686900760421832 5498493658321124600506947888 3515430371713505892127392912 2154618614921181030658724688 1265410932572757113244012912 711793649572175876199757263 383273503615787010261407757 197443926105102399225573693 97248500917438495140954207 45764000431735762419272568 20560637875127661376774632 8811701946483283447189128 3599145865465003098147672 1399667836569723427057428 517685364210719623706172 181889452290252840761628 60629817430084280253876 19146258135816088501224 5719012170438571889976 1613054714739084379224 428786696323047746376 107196674080761936594 25144898858450330806 5519611944537877494 1130522928399324306 215337700647490344 38000770702498296 6186171974825304 924370524973896 126050526132804 15579278510796 1731030945644 171200862756 14887031544 1120529256 71523144 3764376 156849 4851 99 1);
                my @pascal_row100 =
                  qw(1 100 4950 161700 3921225 75287520 1192052400 16007560800 186087894300 1902231808400 17310309456440 141629804643600 1050421051106700 7110542499799200 44186942677323600 253338471349988640 1345860629046814650 6650134872937201800 30664510802988208300 132341572939212267400 535983370403809682970 2041841411062132125600 7332066885177656269200 24865270306254660391200 79776075565900368755100 242519269720337121015504 699574816500972464467800 1917353200780443050763600 4998813702034726525205100 12410847811948286545336800 29372339821610944823963760 66324638306863423796047200 143012501349174257560226775 294692427022540894366527900 580717429720889409486981450 1095067153187962886461165020 1977204582144932989443770175 3420029547493938143902737600 5670048986634686922786117600 9013924030034630492634340800 13746234145802811501267369720 20116440213369968050635175200 28258808871162574166368460400 38116532895986727945334202400 49378235797073715747364762200 61448471214136179596720592960 73470998190814997343905056800 84413487283064039501507937600 93206558875049876949581681100 98913082887808032681188722800 100891344545564193334812497256 98913082887808032681188722800 93206558875049876949581681100 84413487283064039501507937600 73470998190814997343905056800 61448471214136179596720592960 49378235797073715747364762200 38116532895986727945334202400 28258808871162574166368460400 20116440213369968050635175200 13746234145802811501267369720 9013924030034630492634340800 5670048986634686922786117600 3420029547493938143902737600 1977204582144932989443770175 1095067153187962886461165020 580717429720889409486981450 294692427022540894366527900 143012501349174257560226775 66324638306863423796047200 29372339821610944823963760 12410847811948286545336800 4998813702034726525205100 1917353200780443050763600 699574816500972464467800 242519269720337121015504 79776075565900368755100 24865270306254660391200 7332066885177656269200 2041841411062132125600 535983370403809682970 132341572939212267400 30664510802988208300 6650134872937201800 1345860629046814650 253338471349988640 44186942677323600 7110542499799200 1050421051106700 141629804643600 17310309456440 1902231808400 186087894300 16007560800 1192052400 75287520 3921225 161700 4950 100 1);
                my @pascal_row101 =
                  qw(1 101 5050 166650 4082925 79208745 1267339920 17199613200 202095455100 2088319702700 19212541264840 158940114100040 1192050855750300 8160963550905900 51297485177122800 297525414027312240 1599199100396803290 7995995501984016450 37314645675925410100 163006083742200475700 668324943343021950370 2577824781465941808570 9373908296239788394800 32197337191432316660400 104641345872155029146300 322295345286237489770604 942094086221309585483304 2616928017281415515231400 6916166902815169575968700 17409661513983013070541900 41783187633559231369300560 95696978128474368620010960 209337139656037681356273975 437704928371715151926754675 875409856743430303853509350 1675784582908852295948146470 3072271735332895875904935195 5397234129638871133346507775 9090078534128625066688855200 14683973016669317415420458400 22760158175837441993901710520 33862674359172779551902544920 48375249084532542217003635600 66375341767149302111702662800 87494768693060443692698964600 110826707011209895344085355160 134919469404951176940625649760 157884485473879036845412994400 177620046158113916451089618700 192119641762857909630770403900 199804427433372226016001220056 199804427433372226016001220056 192119641762857909630770403900 177620046158113916451089618700 157884485473879036845412994400 134919469404951176940625649760 110826707011209895344085355160 87494768693060443692698964600 66375341767149302111702662800 48375249084532542217003635600 33862674359172779551902544920 22760158175837441993901710520 14683973016669317415420458400 9090078534128625066688855200 5397234129638871133346507775 3072271735332895875904935195 1675784582908852295948146470 875409856743430303853509350 437704928371715151926754675 209337139656037681356273975 95696978128474368620010960 41783187633559231369300560 17409661513983013070541900 6916166902815169575968700 2616928017281415515231400 942094086221309585483304 322295345286237489770604 104641345872155029146300 32197337191432316660400 9373908296239788394800 2577824781465941808570 668324943343021950370 163006083742200475700 37314645675925410100 7995995501984016450 1599199100396803290 297525414027312240 51297485177122800 8160963550905900 1192050855750300 158940114100040 19212541264840 2088319702700 202095455100 17199613200 1267339920 79208745 4082925 166650 5050 101 1);
                my @pascal_row102 =
                  qw(1 102 5151 171700 4249575 83291670 1346548665 18466953120 219295068300 2290415157800 21300860967540 178152655364880 1350990969850340 9353014406656200 59458448728028700 348822899204435040 1896724514424115530 9595194602380819740 45310641177909426550 200320729418125885800 831331027085222426070 3246149724808963758940 11951733077705730203370 41571245487672105055200 136838683063587345806700 426936691158392518916904 1264389431507547075253908 3559022103502725100714704 9533094920096585091200100 24325828416798182646510600 59192849147542244439842460 137480165762033599989311520 305034117784512049976284935 647042068027752833283028650 1313114785115145455780264025 2551194439652282599801655820 4748056318241748171853081665 8469505864971767009251442970 14487312663767496200035362975 23774051550797942482109313600 37444131192506759409322168920 56622832535010221545804255440 82237923443705321768906180520 114750590851681844328706298400 153870110460209745804401627400 198321475704270339036784319760 245746176416161072284711004920 292803954878830213786038644160 335504531631992953296502613100 369739687920971826081860022600 391924069196230135646771623956 399608854866744452032002440112 391924069196230135646771623956 369739687920971826081860022600 335504531631992953296502613100 292803954878830213786038644160 245746176416161072284711004920 198321475704270339036784319760 153870110460209745804401627400 114750590851681844328706298400 82237923443705321768906180520 56622832535010221545804255440 37444131192506759409322168920 23774051550797942482109313600 14487312663767496200035362975 8469505864971767009251442970 4748056318241748171853081665 2551194439652282599801655820 1313114785115145455780264025 647042068027752833283028650 305034117784512049976284935 137480165762033599989311520 59192849147542244439842460 24325828416798182646510600 9533094920096585091200100 3559022103502725100714704 1264389431507547075253908 426936691158392518916904 136838683063587345806700 41571245487672105055200 11951733077705730203370 3246149724808963758940 831331027085222426070 200320729418125885800 45310641177909426550 9595194602380819740 1896724514424115530 348822899204435040 59458448728028700 9353014406656200 1350990969850340 178152655364880 21300860967540 2290415157800 219295068300 18466953120 1346548665 83291670 4249575 171700 5151 102 1);
                my @pascal_row103 =
                  qw(1 103 5253 176851 4421275 87541245 1429840335 19813501785 237762021420 2509710226100 23591276125340 199453516332420 1529143625215220 10704005376506540 68811463134684900 408281347932463740 2245547413628550570 11491919116804935270 54905835780290246290 245631370596035312350 1031651756503348311870 4077480751894186185010 15197882802514693962310 53522978565377835258570 178409928551259450861900 563775374221979864723604 1691326122665939594170812 4823411535010272175968612 13092117023599310191914804 33858923336894767737710700 83518677564340427086353060 196673014909575844429153980 442514283546545649965596455 952076185812264883259313585 1960156853142898289063292675 3864309224767428055581919845 7299250757894030771654737485 13217562183213515181104524635 22956818528739263209286805945 38261364214565438682144676575 61218182743304701891431482520 94066963727516980955126424360 138860755978715543314710435960 196988514295387166097612478920 268620701311891590133107925800 352191586164480084841185947160 444067652120431411321495324680 538550131294991286070749649080 628308486510823167082541257260 705244219552964779378362635700 761663757117201961728631646556 791532924062974587678774064068 791532924062974587678774064068 761663757117201961728631646556 705244219552964779378362635700 628308486510823167082541257260 538550131294991286070749649080 444067652120431411321495324680 352191586164480084841185947160 268620701311891590133107925800 196988514295387166097612478920 138860755978715543314710435960 94066963727516980955126424360 61218182743304701891431482520 38261364214565438682144676575 22956818528739263209286805945 13217562183213515181104524635 7299250757894030771654737485 3864309224767428055581919845 1960156853142898289063292675 952076185812264883259313585 442514283546545649965596455 196673014909575844429153980 83518677564340427086353060 33858923336894767737710700 13092117023599310191914804 4823411535010272175968612 1691326122665939594170812 563775374221979864723604 178409928551259450861900 53522978565377835258570 15197882802514693962310 4077480751894186185010 1031651756503348311870 245631370596035312350 54905835780290246290 11491919116804935270 2245547413628550570 408281347932463740 68811463134684900 10704005376506540 1529143625215220 199453516332420 23591276125340 2509710226100 237762021420 19813501785 1429840335 87541245 4421275 176851 5253 103 1);
                my @pascal_row104 =
                  qw(1 104 5356 182104 4598126 91962520 1517381580 21243342120 257575523205 2747472247520 26100986351440 223044792457760 1728597141547640 12233149001721760 79515468511191440 477092811067148640 2653828761561014310 13737466530433485840 66397754897095181560 300537206376325558640 1277283127099383624220 5109132508397534496880 19275363554408880147320 68720861367892529220880 231932907116637286120470 742185302773239315585504 2255101496887919458894416 6514737657676211770139424 17915528558609582367883416 46951040360494077929625504 117377600901235194824063760 280191692473916271515507040 639187298456121494394750435 1394590469358810533224910040 2912233038955163172322606260 5824466077910326344645212520 11163559982661458827236657330 20516812941107545952759262120 36174380711952778390391330580 61218182743304701891431482520 99479546957870140573576159095 155285146470821682846557906880 232927719706232524269836860320 335849270274102709412322914880 465609215607278756230720404720 620812287476371674974293872960 796259238284911496162681271840 982617783415422697392244973760 1166858617805814453153290906340 1333552706063787946460903892960 1466907976670166741106994282256 1553196681180176549407405710624 1583065848125949175357548128136 1553196681180176549407405710624 1466907976670166741106994282256 1333552706063787946460903892960 1166858617805814453153290906340 982617783415422697392244973760 796259238284911496162681271840 620812287476371674974293872960 465609215607278756230720404720 335849270274102709412322914880 232927719706232524269836860320 155285146470821682846557906880 99479546957870140573576159095 61218182743304701891431482520 36174380711952778390391330580 20516812941107545952759262120 11163559982661458827236657330 5824466077910326344645212520 2912233038955163172322606260 1394590469358810533224910040 639187298456121494394750435 280191692473916271515507040 117377600901235194824063760 46951040360494077929625504 17915528558609582367883416 6514737657676211770139424 2255101496887919458894416 742185302773239315585504 231932907116637286120470 68720861367892529220880 19275363554408880147320 5109132508397534496880 1277283127099383624220 300537206376325558640 66397754897095181560 13737466530433485840 2653828761561014310 477092811067148640 79515468511191440 12233149001721760 1728597141547640 223044792457760 26100986351440 2747472247520 257575523205 21243342120 1517381580 91962520 4598126 182104 5356 104 1);
                my @pascal_row105 =
                  qw(1 105 5460 187460 4780230 96560646 1609344100 22760723700 278818865325 3005047770725 28848458598960 249145778809200 1951641934005400 13961746143269400 91748617512913200 556608279578340080 3130921572628162950 16391295291994500150 80135221427528667400 366934961273420740200 1577820333475709182860 6386415635496918121100 24384496062806414644200 87996224922301409368200 300653768484529815341350 974118209889876601705974 2997286799661158774479920 8769839154564131229033840 24430266216285794138022840 64866568919103660297508920 164328641261729272753689264 397569293375151466339570800 919378990930037765910257475 2033777767814932027619660475 4306823508313973705547516300 8736699116865489516967818780 16988026060571785171881869850 31680372923769004779995919450 56691193653060324343150592700 97392563455257480281822813100 160697729701174842465007641615 254764693428691823420134065975 388212866177054207116394767200 568776989980335233682159775200 801458485881381465643043319600 1086421503083650431205014277680 1417071525761283171136975144800 1778877021700334193554926245600 2149476401221237150545535880100 2500411323869602399614194799300 2800460682733954687567898175216 3020104657850343290514399992880 3136262529306125724764953838760 3136262529306125724764953838760 3020104657850343290514399992880 2800460682733954687567898175216 2500411323869602399614194799300 2149476401221237150545535880100 1778877021700334193554926245600 1417071525761283171136975144800 1086421503083650431205014277680 801458485881381465643043319600 568776989980335233682159775200 388212866177054207116394767200 254764693428691823420134065975 160697729701174842465007641615 97392563455257480281822813100 56691193653060324343150592700 31680372923769004779995919450 16988026060571785171881869850 8736699116865489516967818780 4306823508313973705547516300 2033777767814932027619660475 919378990930037765910257475 397569293375151466339570800 164328641261729272753689264 64866568919103660297508920 24430266216285794138022840 8769839154564131229033840 2997286799661158774479920 974118209889876601705974 300653768484529815341350 87996224922301409368200 24384496062806414644200 6386415635496918121100 1577820333475709182860 366934961273420740200 80135221427528667400 16391295291994500150 3130921572628162950 556608279578340080 91748617512913200 13961746143269400 1951641934005400 249145778809200 28848458598960 3005047770725 278818865325 22760723700 1609344100 96560646 4780230 187460 5460 105 1);
                my @pascal_row106 =
                  qw(1 106 5565 192920 4967690 101340876 1705904746 24370067800 301579589025 3283866636050 31853506369685 277994237408160 2200787712814600 15913388077274800 105710363656182600 648356897091253280 3687529852206503030 19522216864622663100 96526516719523167550 447070182700949407600 1944755294749129923060 7964235968972627303960 30770911698303332765300 112380720985107824012400 388649993406831224709550 1274771978374406417047324 3971405009551035376185894 11767125954225290003513760 33200105370849925367056680 89296835135389454435531760 229195210180832933051198184 561897934636880739093260064 1316948284305189232249828275 2953156758744969793529917950 6340601276128905733167176775 13043522625179463222515335080 25724725177437274688849688630 48668398984340789951877789300 88371566576829329123146512150 154083757108317804624973405800 258090293156432322746830454715 415462423129866665885141707590 642977559605746030536528833175 956989856157389440798554542400 1370235475861716699325203094800 1887879988965031896848057597280 2503493028844933602341989422480 3195948547461617364691901390400 3928353422921571344100462125700 4649887725090839550159730679400 5300872006603557087182092974516 5820565340584297978082298168096 6156367187156469015279353831640 6272525058612251449529907677520 6156367187156469015279353831640 5820565340584297978082298168096 5300872006603557087182092974516 4649887725090839550159730679400 3928353422921571344100462125700 3195948547461617364691901390400 2503493028844933602341989422480 1887879988965031896848057597280 1370235475861716699325203094800 956989856157389440798554542400 642977559605746030536528833175 415462423129866665885141707590 258090293156432322746830454715 154083757108317804624973405800 88371566576829329123146512150 48668398984340789951877789300 25724725177437274688849688630 13043522625179463222515335080 6340601276128905733167176775 2953156758744969793529917950 1316948284305189232249828275 561897934636880739093260064 229195210180832933051198184 89296835135389454435531760 33200105370849925367056680 11767125954225290003513760 3971405009551035376185894 1274771978374406417047324 388649993406831224709550 112380720985107824012400 30770911698303332765300 7964235968972627303960 1944755294749129923060 447070182700949407600 96526516719523167550 19522216864622663100 3687529852206503030 648356897091253280 105710363656182600 15913388077274800 2200787712814600 277994237408160 31853506369685 3283866636050 301579589025 24370067800 1705904746 101340876 4967690 192920 5565 106 1);
                my @pascal_row107 =
                  qw(1 107 5671 198485 5160610 106308566 1807245622 26075972546 325949656825 3585446225075 35137373005735 309847743777845 2478781950222760 18114175790089400 121623751733457400 754067260747435880 4335886749297756310 23209746716829166130 116048733584145830650 543596699420472575150 2391825477450079330660 9908991263721757227020 38735147667275960069260 143151632683411156777700 501030714391939048721950 1663421971781237641756874 5246176987925441793233218 15738530963776325379699654 44967231325075215370570440 122496940506239379802588440 318492045316222387486729944 791093144817713672144458248 1878846218942069971343088339 4270105043050159025779746225 9293758034873875526697094725 19384123901308368955682511855 38768247802616737911365023710 74393124161778064640727477930 137039965561170119075024301450 242455323685147133748119917950 412174050264750127371803860515 673552716286298988631972162305 1058439982735612696421670540765 1599967415763135471335083375575 2327225332019106140123757637200 3258115464826748596173260692080 4391373017809965499190047019760 5699441576306550967033890812880 7124301970383188708792363516100 8578241148012410894260192805100 9950759731694396637341823653916 11121437347187855065264391142612 11976932527740766993361651999736 12428892245768720464809261509160 12428892245768720464809261509160 11976932527740766993361651999736 11121437347187855065264391142612 9950759731694396637341823653916 8578241148012410894260192805100 7124301970383188708792363516100 5699441576306550967033890812880 4391373017809965499190047019760 3258115464826748596173260692080 2327225332019106140123757637200 1599967415763135471335083375575 1058439982735612696421670540765 673552716286298988631972162305 412174050264750127371803860515 242455323685147133748119917950 137039965561170119075024301450 74393124161778064640727477930 38768247802616737911365023710 19384123901308368955682511855 9293758034873875526697094725 4270105043050159025779746225 1878846218942069971343088339 791093144817713672144458248 318492045316222387486729944 122496940506239379802588440 44967231325075215370570440 15738530963776325379699654 5246176987925441793233218 1663421971781237641756874 501030714391939048721950 143151632683411156777700 38735147667275960069260 9908991263721757227020 2391825477450079330660 543596699420472575150 116048733584145830650 23209746716829166130 4335886749297756310 754067260747435880 121623751733457400 18114175790089400 2478781950222760 309847743777845 35137373005735 3585446225075 325949656825 26075972546 1807245622 106308566 5160610 198485 5671 107 1);
                my @pascal_row108 =
                  qw(1 108 5778 204156 5359095 111469176 1913554188 27883218168 352025629371 3911395881900 38722819230810 344985116783580 2788629694000605 20592957740312160 139737927523546800 875691012480893280 5089954010045192190 27545633466126922440 139258480300974996780 659645433004618405800 2935422176870551905810 12300816741171836557680 48644138930997717296280 181886780350687116846960 644182347075350205499650 2164452686173176690478824 6909598959706679434990092 20984707951701767172932872 60705762288851540750270094 167464171831314595173158880 440988985822461767289318384 1109585190133936059631188192 2669939363759783643487546587 6148951261992228997122834564 13563863077924034552476840950 28677881936182244482379606580 58152371703925106867047535565 113161371964394802552092501640 211433089722948183715751779380 379495289246317252823144219400 654629373949897261119923778465 1085726766551049116003776022820 1731992699021911685053642703070 2658407398498748167756753916340 3927192747782241611458841012775 5585340796845854736297018329280 7649488482636714095363307711840 10090814594116516466223937832640 12823743546689739675826254328980 15702543118395599603052556321200 18529000879706807531602016459016 21072197078882251702606214796528 23098369874928622058626043142348 24405824773509487458170913508896 24857784491537440929618523018320 24405824773509487458170913508896 23098369874928622058626043142348 21072197078882251702606214796528 18529000879706807531602016459016 15702543118395599603052556321200 12823743546689739675826254328980 10090814594116516466223937832640 7649488482636714095363307711840 5585340796845854736297018329280 3927192747782241611458841012775 2658407398498748167756753916340 1731992699021911685053642703070 1085726766551049116003776022820 654629373949897261119923778465 379495289246317252823144219400 211433089722948183715751779380 113161371964394802552092501640 58152371703925106867047535565 28677881936182244482379606580 13563863077924034552476840950 6148951261992228997122834564 2669939363759783643487546587 1109585190133936059631188192 440988985822461767289318384 167464171831314595173158880 60705762288851540750270094 20984707951701767172932872 6909598959706679434990092 2164452686173176690478824 644182347075350205499650 181886780350687116846960 48644138930997717296280 12300816741171836557680 2935422176870551905810 659645433004618405800 139258480300974996780 27545633466126922440 5089954010045192190 875691012480893280 139737927523546800 20592957740312160 2788629694000605 344985116783580 38722819230810 3911395881900 352025629371 27883218168 1913554188 111469176 5359095 204156 5778 108 1);
                my @pascal_row109 =
                  qw(1 109 5886 209934 5563251 116828271 2025023364 29796772356 379908847539 4263421511271 42634215112710 383707936014390 3133614810784185 23381587434312765 160330885263858960 1015428940004440080 5965645022526085470 32635587476172114630 166804113767101919220 798903913305593402580 3595067609875170311610 15236238918042388463490 60944955672169553853960 230530919281684834143240 826069127426037322346610 2808635033248526895978474 9074051645879856125468916 27894306911408446607922964 81690470240553307923202966 228169934120166135923428974 608453157653776362462477264 1550574175956397826920506576 3779524553893719703118734779 8818890625752012640610381151 19712814339916263549599675514 42241745014106279034856447530 86830253640107351349427142145 171313743668319909419140037205 324594461687342986267844281020 590928378969265436538895998780 1034124663196214513943067997865 1740356140500946377123699801285 2817719465572960801057418725890 4390400097520659852810396619410 6585600146280989779215594929115 9512533544628096347755859342055 13234829279482568831660326041120 17740303076753230561587245544480 22914558140806256142050192161620 28526286665085339278878810650180 34231543998102407134654572780216 39601197958589059234208231255544 44170566953810873761232257938876 47504194648438109516796956651244 49263609265046928387789436527216 49263609265046928387789436527216 47504194648438109516796956651244 44170566953810873761232257938876 39601197958589059234208231255544 34231543998102407134654572780216 28526286665085339278878810650180 22914558140806256142050192161620 17740303076753230561587245544480 13234829279482568831660326041120 9512533544628096347755859342055 6585600146280989779215594929115 4390400097520659852810396619410 2817719465572960801057418725890 1740356140500946377123699801285 1034124663196214513943067997865 590928378969265436538895998780 324594461687342986267844281020 171313743668319909419140037205 86830253640107351349427142145 42241745014106279034856447530 19712814339916263549599675514 8818890625752012640610381151 3779524553893719703118734779 1550574175956397826920506576 608453157653776362462477264 228169934120166135923428974 81690470240553307923202966 27894306911408446607922964 9074051645879856125468916 2808635033248526895978474 826069127426037322346610 230530919281684834143240 60944955672169553853960 15236238918042388463490 3595067609875170311610 798903913305593402580 166804113767101919220 32635587476172114630 5965645022526085470 1015428940004440080 160330885263858960 23381587434312765 3133614810784185 383707936014390 42634215112710 4263421511271 379908847539 29796772356 2025023364 116828271 5563251 209934 5886 109 1);
                my @pascal_row110 =
                  qw(1 110 5995 215820 5773185 122391522 2141851635 31821795720 409705619895 4643330358810 46897636623981 426342151127100 3517322746798575 26515202245096950 183712472698171725 1175759825268299040 6981073962530525550 38601232498698200100 199439701243274033850 965708027072695321800 4393971523180763714190 18831306527917558775100 76181194590211942317450 291475874953854387997200 1056600046707722156489850 3634704160674564218325084 11882686679128383021447390 36968358557288302733391880 109584777151961754531125930 309860404360719443846631940 836623091773942498385906238 2159027333610174189382983840 5330098729850117530039241355 12598415179645732343729115930 28531704965668276190210056665 61954559354022542584456123044 129071998654213630384283589675 258143997308427260768567179350 495908205355662895686984318225 915522840656608422806740279800 1625053042165479950481963996645 2774480803697160891066767799150 4558075606073907178181118527175 7208119563093620653867815345300 10976000243801649632025991548525 16098133690909086126971454271170 22747362824110665179416185383175 30975132356235799393247571585600 40654861217559486703637437706100 51440844805891595420929002811800 62757830663187746413533383430396 73832741956691466368862804035760 83771764912399932995440489194420 91674761602248983278029214590120 96767803913485037904586393178460 98527218530093856775578873054432 96767803913485037904586393178460 91674761602248983278029214590120 83771764912399932995440489194420 73832741956691466368862804035760 62757830663187746413533383430396 51440844805891595420929002811800 40654861217559486703637437706100 30975132356235799393247571585600 22747362824110665179416185383175 16098133690909086126971454271170 10976000243801649632025991548525 7208119563093620653867815345300 4558075606073907178181118527175 2774480803697160891066767799150 1625053042165479950481963996645 915522840656608422806740279800 495908205355662895686984318225 258143997308427260768567179350 129071998654213630384283589675 61954559354022542584456123044 28531704965668276190210056665 12598415179645732343729115930 5330098729850117530039241355 2159027333610174189382983840 836623091773942498385906238 309860404360719443846631940 109584777151961754531125930 36968358557288302733391880 11882686679128383021447390 3634704160674564218325084 1056600046707722156489850 291475874953854387997200 76181194590211942317450 18831306527917558775100 4393971523180763714190 965708027072695321800 199439701243274033850 38601232498698200100 6981073962530525550 1175759825268299040 183712472698171725 26515202245096950 3517322746798575 426342151127100 46897636623981 4643330358810 409705619895 31821795720 2141851635 122391522 5773185 215820 5995 110 1);
                my @pascal_row111 =
                  qw(1 111 6105 221815 5989005 128164707 2264243157 33963647355 441527415615 5053035978705 51540966982791 473239787751081 3943664897925675 30032524991895525 210227674943268675 1359472297966470765 8156833787798824590 45582306461228725650 238040933741972233950 1165147728315969355650 5359679550253459035990 23225278051098322489290 95012501118129501092550 367657069544066330314650 1348075921661576544487050 4691304207382286374814934 15517390839802947239772474 48851045236416685754839270 146553135709250057264517810 419445181512681198377757870 1146483496134661942232538178 2995650425384116687768890078 7489126063460291719422225195 17928513909495849873768357285 41130120145314008533939172595 90486264319690818774666179709 191026558008236172968739712719 387215995962640891152850769025 754052202664090156455551497575 1411431046012271318493724598025 2540575882822088373288704276445 4399533845862640841548731795795 7332556409771068069247886326325 11766195169167527832048933872475 18184119806895270285893806893825 27074133934710735758997445819695 38845496515019751306387639654345 53722495180346464572663756968775 71629993573795286096885009291700 92095706023451082124566440517900 114198675469079341834462386242196 136590572619879212782396187466156 157604506869091399364303293230180 175446526514648916273469703784540 188442565515734021182615607768580 195295022443578894680165266232892 195295022443578894680165266232892 188442565515734021182615607768580 175446526514648916273469703784540 157604506869091399364303293230180 136590572619879212782396187466156 114198675469079341834462386242196 92095706023451082124566440517900 71629993573795286096885009291700 53722495180346464572663756968775 38845496515019751306387639654345 27074133934710735758997445819695 18184119806895270285893806893825 11766195169167527832048933872475 7332556409771068069247886326325 4399533845862640841548731795795 2540575882822088373288704276445 1411431046012271318493724598025 754052202664090156455551497575 387215995962640891152850769025 191026558008236172968739712719 90486264319690818774666179709 41130120145314008533939172595 17928513909495849873768357285 7489126063460291719422225195 2995650425384116687768890078 1146483496134661942232538178 419445181512681198377757870 146553135709250057264517810 48851045236416685754839270 15517390839802947239772474 4691304207382286374814934 1348075921661576544487050 367657069544066330314650 95012501118129501092550 23225278051098322489290 5359679550253459035990 1165147728315969355650 238040933741972233950 45582306461228725650 8156833787798824590 1359472297966470765 210227674943268675 30032524991895525 3943664897925675 473239787751081 51540966982791 5053035978705 441527415615 33963647355 2264243157 128164707 5989005 221815 6105 111 1);
                my @pascal_row112 =
                  qw(1 112 6216 227920 6210820 134153712 2392407864 36227890512 475491062970 5494563394320 56594002961496 524780754733872 4416904685676756 33976189889821200 240260199935164200 1569699972909739440 9516306085765295355 53739140249027550240 283623240203200959600 1403188662057941589600 6524827278569428391640 28584957601351781525280 118237779169227823581840 462669570662195831407200 1715732991205642874801700 6039380129043862919301984 20208695047185233614587408 64368436076219632994611744 195404180945666743019357080 565998317221931255642275680 1565928677647343140610296048 4142133921518778630001428256 10484776488844408407191115273 25417639972956141593190582480 59058634054809858407707529880 131616384465004827308605352304 281512822327926991743405892428 578242553970877064121590481744 1141268198626731047608402266600 2165483248676361474949276095600 3952006928834359691782428874470 6940109728684729214837436072240 11732090255633708910796618122120 19098751578938595901296820198800 29950314976062798117942740766300 45258253741606006044891252713520 65919630449730487065385085474040 92567991695366215879051396623120 125352488754141750669548766260475 163725699597246368221451449809600 206294381492530423959028826760096 250789248088958554616858573708352 294195079488970612146699480696336 333051033383740315637772997014720 363889092030382937456085311553120 383737587959312915862780874001472 390590044887157789360330532465784 383737587959312915862780874001472 363889092030382937456085311553120 333051033383740315637772997014720 294195079488970612146699480696336 250789248088958554616858573708352 206294381492530423959028826760096 163725699597246368221451449809600 125352488754141750669548766260475 92567991695366215879051396623120 65919630449730487065385085474040 45258253741606006044891252713520 29950314976062798117942740766300 19098751578938595901296820198800 11732090255633708910796618122120 6940109728684729214837436072240 3952006928834359691782428874470 2165483248676361474949276095600 1141268198626731047608402266600 578242553970877064121590481744 281512822327926991743405892428 131616384465004827308605352304 59058634054809858407707529880 25417639972956141593190582480 10484776488844408407191115273 4142133921518778630001428256 1565928677647343140610296048 565998317221931255642275680 195404180945666743019357080 64368436076219632994611744 20208695047185233614587408 6039380129043862919301984 1715732991205642874801700 462669570662195831407200 118237779169227823581840 28584957601351781525280 6524827278569428391640 1403188662057941589600 283623240203200959600 53739140249027550240 9516306085765295355 1569699972909739440 240260199935164200 33976189889821200 4416904685676756 524780754733872 56594002961496 5494563394320 475491062970 36227890512 2392407864 134153712 6210820 227920 6216 112 1);
                my @pascal_row113 =
                  qw(1 113 6328 234136 6438740 140364532 2526561576 38620298376 511718953482 5970054457290 62088566355816 581374757695368 4941685440410628 38393094575497956 274236389824985400 1809960172844903640 11086006058675034795 63255446334792845595 337362380452228509840 1686811902261142549200 7928015940627369981240 35109784879921209916920 146822736770579605107120 580907349831423654989040 2178402561867838706208900 7755113120249505794103684 26248075176229096533889392 84577131123404866609199152 259772617021886376013968824 761402498167597998661632760 2131926994869274396252571728 5708062599166121770611724304 14626910410363187037192543529 35902416461800550000381697753 84476274027766000000898112360 190675018519814685716312882184 413129206792931819052011244732 859755376298804055864996374172 1719510752597608111729992748344 3306751447303092522557678362200 6117490177510721166731704970070 10892116657519088906619864946710 18672199984318438125634054194360 30830841834572304812093438320920 49049066555001394019239560965100 75208568717668804162833993479820 111177884191336493110276338187560 158487622145096702944436482097160 217920480449507966548600162883595 289078188351388118891000216070075 370020081089776792180480276569696 457083629581488978575887400468448 544984327577929166763558054404688 627246112872710927784472477711056 696940125414123253093858308567840 747626679989695853318866185554592 774327632846470705223111406467256 774327632846470705223111406467256 747626679989695853318866185554592 696940125414123253093858308567840 627246112872710927784472477711056 544984327577929166763558054404688 457083629581488978575887400468448 370020081089776792180480276569696 289078188351388118891000216070075 217920480449507966548600162883595 158487622145096702944436482097160 111177884191336493110276338187560 75208568717668804162833993479820 49049066555001394019239560965100 30830841834572304812093438320920 18672199984318438125634054194360 10892116657519088906619864946710 6117490177510721166731704970070 3306751447303092522557678362200 1719510752597608111729992748344 859755376298804055864996374172 413129206792931819052011244732 190675018519814685716312882184 84476274027766000000898112360 35902416461800550000381697753 14626910410363187037192543529 5708062599166121770611724304 2131926994869274396252571728 761402498167597998661632760 259772617021886376013968824 84577131123404866609199152 26248075176229096533889392 7755113120249505794103684 2178402561867838706208900 580907349831423654989040 146822736770579605107120 35109784879921209916920 7928015940627369981240 1686811902261142549200 337362380452228509840 63255446334792845595 11086006058675034795 1809960172844903640 274236389824985400 38393094575497956 4941685440410628 581374757695368 62088566355816 5970054457290 511718953482 38620298376 2526561576 140364532 6438740 234136 6328 113 1);
                my @pascal_row114 =
                  qw(1 114 6441 240464 6672876 146803272 2666926108 41146859952 550339251858 6481773410772 68058620813106 643463324051184 5523060198105996 43334780015908584 312629484400483356 2084196562669889040 12895966231519938435 74341452393467880390 400617826787021355435 2024174282713371059040 9614827842888512530440 43037800820548579898160 181932521650500815024040 727730086602003260096160 2759309911699262361197940 9933515682117344500312584 34003188296478602327993076 110825206299633963143088544 344349748145291242623167976 1021175115189484374675601584 2893329493036872394914204488 7839989594035396166864296032 20334973009529308807804267833 50529326872163737037574241282 120378690489566550001279810113 275151292547580685717210994544 603804225312746504768324126916 1272884583091735874917007618904 2579266128896412167594989122516 5026262199900700634287671110544 9424241624813813689289383332270 17009606835029810073351569916780 29564316641837527032253919141070 49503041818890742937727492515280 79879908389573698831332999286020 124257635272670198182073554444920 186386452909005297273110331667380 269665506336433196054712820284720 376408102594604669493036644980755 506998668800896085439600378953670 659098269441164911071480492639771 827103710671265770756367677038144 1002067957159418145339445454873136 1172230440450640094548030532115744 1324186238286834180878330786278896 1444566805403819106412724494122432 1521954312836166558541977592021848 1548655265692941410446222812934512 1521954312836166558541977592021848 1444566805403819106412724494122432 1324186238286834180878330786278896 1172230440450640094548030532115744 1002067957159418145339445454873136 827103710671265770756367677038144 659098269441164911071480492639771 506998668800896085439600378953670 376408102594604669493036644980755 269665506336433196054712820284720 186386452909005297273110331667380 124257635272670198182073554444920 79879908389573698831332999286020 49503041818890742937727492515280 29564316641837527032253919141070 17009606835029810073351569916780 9424241624813813689289383332270 5026262199900700634287671110544 2579266128896412167594989122516 1272884583091735874917007618904 603804225312746504768324126916 275151292547580685717210994544 120378690489566550001279810113 50529326872163737037574241282 20334973009529308807804267833 7839989594035396166864296032 2893329493036872394914204488 1021175115189484374675601584 344349748145291242623167976 110825206299633963143088544 34003188296478602327993076 9933515682117344500312584 2759309911699262361197940 727730086602003260096160 181932521650500815024040 43037800820548579898160 9614827842888512530440 2024174282713371059040 400617826787021355435 74341452393467880390 12895966231519938435 2084196562669889040 312629484400483356 43334780015908584 5523060198105996 643463324051184 68058620813106 6481773410772 550339251858 41146859952 2666926108 146803272 6672876 240464 6441 114 1);
                my @pascal_row115 =
                  qw(1 115 6555 246905 6913340 153476148 2813729380 43813786060 591486111810 7032112662630 74540394223878 711521944864290 6166523522157180 48857840214014580 355964264416391940 2396826047070372396 14980162794189827475 87237418624987818825 474959279180489235825 2424792109500392414475 11639002125601883589480 52652628663437092428600 224970322471049394922200 909662608252504075120200 3487039998301265621294100 12692825593816606861510524 43936703978595946828305660 144828394596112565471081620 455174954444925205766256520 1365524863334775617298769560 3914504608226356769589806072 10733319087072268561778500520 28174962603564704974668563865 70864299881693045845378509115 170908017361730287038854051395 395529983037147235718490804657 878955517860327190485535121460 1876688808404482379685331745820 3852150711988148042511996741420 7605528328797112801882660233060 14450503824714514323577054442814 26433848459843623762640953249050 46573923476867337105605489057850 79067358460728269969981411656350 129382950208464441769060491801300 204137543662243897013406553730940 310644088181675495455183886112300 456051959245438493327823151952100 646073608931037865547749465265475 883406771395500754932637023934425 1166096938242060996511080871593441 1486201980112430681827848169677915 1829171667830683916095813131911280 2174298397610058239887475986988880 2496416678737474275426361318394640 2768753043690653287291055280401328 2966521118239985664954702086144280 3070609578529107968988200404956360 3070609578529107968988200404956360 2966521118239985664954702086144280 2768753043690653287291055280401328 2496416678737474275426361318394640 2174298397610058239887475986988880 1829171667830683916095813131911280 1486201980112430681827848169677915 1166096938242060996511080871593441 883406771395500754932637023934425 646073608931037865547749465265475 456051959245438493327823151952100 310644088181675495455183886112300 204137543662243897013406553730940 129382950208464441769060491801300 79067358460728269969981411656350 46573923476867337105605489057850 26433848459843623762640953249050 14450503824714514323577054442814 7605528328797112801882660233060 3852150711988148042511996741420 1876688808404482379685331745820 878955517860327190485535121460 395529983037147235718490804657 170908017361730287038854051395 70864299881693045845378509115 28174962603564704974668563865 10733319087072268561778500520 3914504608226356769589806072 1365524863334775617298769560 455174954444925205766256520 144828394596112565471081620 43936703978595946828305660 12692825593816606861510524 3487039998301265621294100 909662608252504075120200 224970322471049394922200 52652628663437092428600 11639002125601883589480 2424792109500392414475 474959279180489235825 87237418624987818825 14980162794189827475 2396826047070372396 355964264416391940 48857840214014580 6166523522157180 711521944864290 74540394223878 7032112662630 591486111810 43813786060 2813729380 153476148 6913340 246905 6555 115 1);
                my @pascal_row116 =
                  qw(1 116 6670 253460 7160245 160389488 2967205528 46627515440 635299897870 7623598774440 81572506886508 786062339088168 6878045467021470 55024363736171760 404822104630406520 2752790311486764336 17376988841260199871 102217581419177646300 562196697805477054650 2899751388680881650300 14063794235102276003955 64291630789038976018080 277622951134486487350800 1134632930723553470042400 4396702606553769696414300 16179865592117872482804624 56629529572412553689816184 188765098574708512299387280 600003349041037771237338140 1820699817779700823065026080 5280029471561132386888575632 14647823695298625331368306592 38908281690636973536447064385 99039262485257750820047072980 241772317243423332884232560510 566438000398877522757344856052 1274485500897474426204025926117 2755644326264809570170866867280 5728839520392630422197328487240 11457679040785260844394656974480 22056032153511627125459714675874 40884352284558138086218007691864 73007771936710960868246442306900 125641281937595607075586900714200 208450308669192711739041903457650 333520493870708338782467045532240 514781631843919392468590439843240 766696047427113988783007038064400 1102125568176476358875572617217575 1529480380326538620480386489199900 2049503709637561751443717895527866 2652298918354491678338929041271356 3315373647943114597923661301589195 4003470065440742155983289118900160 4670715076347532515313837305383520 5265169722428127562717416598795968 5735274161930638952245757366545608 6037130696769093633942902491100640 6141219157058215937976400809912720 6037130696769093633942902491100640 5735274161930638952245757366545608 5265169722428127562717416598795968 4670715076347532515313837305383520 4003470065440742155983289118900160 3315373647943114597923661301589195 2652298918354491678338929041271356 2049503709637561751443717895527866 1529480380326538620480386489199900 1102125568176476358875572617217575 766696047427113988783007038064400 514781631843919392468590439843240 333520493870708338782467045532240 208450308669192711739041903457650 125641281937595607075586900714200 73007771936710960868246442306900 40884352284558138086218007691864 22056032153511627125459714675874 11457679040785260844394656974480 5728839520392630422197328487240 2755644326264809570170866867280 1274485500897474426204025926117 566438000398877522757344856052 241772317243423332884232560510 99039262485257750820047072980 38908281690636973536447064385 14647823695298625331368306592 5280029471561132386888575632 1820699817779700823065026080 600003349041037771237338140 188765098574708512299387280 56629529572412553689816184 16179865592117872482804624 4396702606553769696414300 1134632930723553470042400 277622951134486487350800 64291630789038976018080 14063794235102276003955 2899751388680881650300 562196697805477054650 102217581419177646300 17376988841260199871 2752790311486764336 404822104630406520 55024363736171760 6878045467021470 786062339088168 81572506886508 7623598774440 635299897870 46627515440 2967205528 160389488 7160245 253460 6670 116 1);
                my @pascal_row117 =
                  qw(1 117 6786 260130 7413705 167549733 3127595016 49594720968 681927413310 8258898672310 89196105660948 867634845974676 7664107806109638 61902409203193230 459846468366578280 3157612416117170856 20129779152746964207 119594570260437846171 664414279224654700950 3461948086486358704950 16963545623783157654255 78355425024141252022035 341914581923525463368880 1412255881858039957393200 5531335537277323166456700 20576568198671642179218924 72809395164530426172620808 245394628147121065989203464 788768447615746283536725420 2420703166820738594302364220 7100729289340833209953601712 19927853166859757718256882224 53556105385935598867815370977 137947544175894724356494137365 340811579728681083704279633490 808210317642300855641577416562 1840923501296351948961370782169 4030129827162283996374892793397 8484483846657439992368195354520 17186518561177891266591985461720 33513711194296887969854371650354 62940384438069765211677722367738 113892124221269098954464449998764 198649053874306567943833343021100 334091590606788318814628804171850 541970802539901050521508948989890 848302125714627731251057485375480 1281477679271033381251597477907640 1868821615603590347658579655281975 2631605948503014979355959106417475 3578984089964100371924104384727766 4701802627992053429782646936799222 5967672566297606276262590342860551 7318843713383856753906950420489355 8674185141788274671297126424283680 9935884798775660078031253904179488 11000443884358766514963173965341576 11772404858699732586188659857646248 12178349853827309571919303301013360 12178349853827309571919303301013360 11772404858699732586188659857646248 11000443884358766514963173965341576 9935884798775660078031253904179488 8674185141788274671297126424283680 7318843713383856753906950420489355 5967672566297606276262590342860551 4701802627992053429782646936799222 3578984089964100371924104384727766 2631605948503014979355959106417475 1868821615603590347658579655281975 1281477679271033381251597477907640 848302125714627731251057485375480 541970802539901050521508948989890 334091590606788318814628804171850 198649053874306567943833343021100 113892124221269098954464449998764 62940384438069765211677722367738 33513711194296887969854371650354 17186518561177891266591985461720 8484483846657439992368195354520 4030129827162283996374892793397 1840923501296351948961370782169 808210317642300855641577416562 340811579728681083704279633490 137947544175894724356494137365 53556105385935598867815370977 19927853166859757718256882224 7100729289340833209953601712 2420703166820738594302364220 788768447615746283536725420 245394628147121065989203464 72809395164530426172620808 20576568198671642179218924 5531335537277323166456700 1412255881858039957393200 341914581923525463368880 78355425024141252022035 16963545623783157654255 3461948086486358704950 664414279224654700950 119594570260437846171 20129779152746964207 3157612416117170856 459846468366578280 61902409203193230 7664107806109638 867634845974676 89196105660948 8258898672310 681927413310 49594720968 3127595016 167549733 7413705 260130 6786 117 1);
                my @pascal_row118 =
                  qw(1 118 6903 266916 7673835 174963438 3295144749 52722315984 731522134278 8940826085620 97455004333258 956830951635624 8531742652084314 69566517009302868 521748877569771510 3617458884483749136 23287391568864135063 139724349413184810378 784008849485092547121 4126362365711013405900 20425493710269516359205 95318970647924409676290 420270006947666715390915 1754170463781565420762080 6943591419135363123849900 26107903735948965345675624 93385963363202068351839732 318204023311651492161824272 1034163075762867349525928884 3209471614436484877839089640 9521432456161571804255965932 27028582456200590928210483936 73483958552795356586072253201 191503649561830323224309508342 478759123904575808060773770855 1149021897370981939345857050052 2649133818938652804602948198731 5871053328458635945336263575566 12514613673819723988743088147917 25671002407835331258960180816240 50700229755474779236446357112074 96454095632366653181532094018092 176832508659338864166142172366502 312541178095575666898297793019864 532740644481094886758462147192950 876062393146689369336137753161740 1390272928254528781772566434365370 2129779804985661112502654963283120 3150299294874623728910177133189615 4500427564106605327014538761699450 6210590038467115351280063491145241 8280786717956153801706751321526988 10669475194289659706045237279659773 13286516279681463030169540763349906 15993028855172131425204076844773035 18610069940563934749328380328463168 20936328683134426592994427869521064 22772848743058499101151833822987824 23950754712527042158107963158659608 24356699707654619143838606602026720 23950754712527042158107963158659608 22772848743058499101151833822987824 20936328683134426592994427869521064 18610069940563934749328380328463168 15993028855172131425204076844773035 13286516279681463030169540763349906 10669475194289659706045237279659773 8280786717956153801706751321526988 6210590038467115351280063491145241 4500427564106605327014538761699450 3150299294874623728910177133189615 2129779804985661112502654963283120 1390272928254528781772566434365370 876062393146689369336137753161740 532740644481094886758462147192950 312541178095575666898297793019864 176832508659338864166142172366502 96454095632366653181532094018092 50700229755474779236446357112074 25671002407835331258960180816240 12514613673819723988743088147917 5871053328458635945336263575566 2649133818938652804602948198731 1149021897370981939345857050052 478759123904575808060773770855 191503649561830323224309508342 73483958552795356586072253201 27028582456200590928210483936 9521432456161571804255965932 3209471614436484877839089640 1034163075762867349525928884 318204023311651492161824272 93385963363202068351839732 26107903735948965345675624 6943591419135363123849900 1754170463781565420762080 420270006947666715390915 95318970647924409676290 20425493710269516359205 4126362365711013405900 784008849485092547121 139724349413184810378 23287391568864135063 3617458884483749136 521748877569771510 69566517009302868 8531742652084314 956830951635624 97455004333258 8940826085620 731522134278 52722315984 3295144749 174963438 7673835 266916 6903 118 1);
                my @pascal_row119 =
                  qw(1 119 7021 273819 7940751 182637273 3470108187 56017460733 784244450262 9672348219898 106395830418878 1054285955968882 9488573603719938 78098259661387182 591315394579074378 4139207762053520646 26904850453347884199 163011740982048945441 923733198898277357499 4910371215196105953021 24551856075980529765105 115744464358193926035495 515588977595591125067205 2174440470729232136152995 8697761882916928544611980 33051495155084328469525524 119493867099151033697515356 411589986674853560513664004 1352367099074518841687753156 4243634690199352227365018524 12730904070598056682095055572 36550014912362162732466449868 100512541008995947514282737137 264987608114625679810381761543 670262773466406131285083279197 1627781021275557747406630820907 3798155716309634743948805248783 8520187147397288749939211774297 18385667002278359934079351723483 38185616081655055247703268964157 76371232163310110495406537928314 147154325387841432417978451130166 273286604291705517347674266384594 489373686754914531064439965386366 845281822576670553656759940212814 1408803037627784256094599900354690 2266335321401218151108704187527110 3520052733240189894275221397648490 5280079099860284841412832096472735 7650726858981229055924715894889065 10711017602573720678294602252844691 14491376756423269152986814812672229 18950261912245813507751988601186761 23955991473971122736214778043009679 29279545134853594455373617608122941 34603098795736066174532457173236203 39546398623698361342322808197984232 43709177426192925694146261692508888 46723603455585541259259796981647432 48307454420181661301946569760686328 48307454420181661301946569760686328 46723603455585541259259796981647432 43709177426192925694146261692508888 39546398623698361342322808197984232 34603098795736066174532457173236203 29279545134853594455373617608122941 23955991473971122736214778043009679 18950261912245813507751988601186761 14491376756423269152986814812672229 10711017602573720678294602252844691 7650726858981229055924715894889065 5280079099860284841412832096472735 3520052733240189894275221397648490 2266335321401218151108704187527110 1408803037627784256094599900354690 845281822576670553656759940212814 489373686754914531064439965386366 273286604291705517347674266384594 147154325387841432417978451130166 76371232163310110495406537928314 38185616081655055247703268964157 18385667002278359934079351723483 8520187147397288749939211774297 3798155716309634743948805248783 1627781021275557747406630820907 670262773466406131285083279197 264987608114625679810381761543 100512541008995947514282737137 36550014912362162732466449868 12730904070598056682095055572 4243634690199352227365018524 1352367099074518841687753156 411589986674853560513664004 119493867099151033697515356 33051495155084328469525524 8697761882916928544611980 2174440470729232136152995 515588977595591125067205 115744464358193926035495 24551856075980529765105 4910371215196105953021 923733198898277357499 163011740982048945441 26904850453347884199 4139207762053520646 591315394579074378 78098259661387182 9488573603719938 1054285955968882 106395830418878 9672348219898 784244450262 56017460733 3470108187 182637273 7940751 273819 7021 119 1);
                my @pascal_row120 =
                  qw(1 120 7140 280840 8214570 190578024 3652745460 59487568920 840261910995 10456592670160 116068178638776 1160681786387760 10542859559688820 87586833265107120 669413654240461560 4730523156632595024 31044058215401404845 189916591435396829640 1086744939880326302940 5834104414094383310520 29462227291176635718126 140296320434174455800600 631333441953785051102700 2690029448324823261220200 10872202353646160680764975 41749257038001257014137504 152545362254235362167040880 531083853774004594211179360 1763957085749372402201417160 5596001789273871069052771680 16974538760797408909460074096 49280918982960219414561505440 137062555921358110246749187005 365500149123621627324664498680 935250381581031811095465040740 2298043794741963878691714100104 5425936737585192491355436069690 12318342863706923493888017023080 26905854149675648684018563497780 56571283083933415181782620687640 114556848244965165743109806892471 223525557551151542913384989058480 420440929679546949765652717514760 762660291046620048412114231770960 1334655509331585084721199905599180 2254084860204454809751359840567504 3675138359029002407203304087881800 5786388054641408045383925585175600 8800131833100474735688053494121225 12930805958841513897337547991361800 18361744461554949734219318147733756 25202394358996989831281417065516920 33441638668669082660738803413858990 42906253386216936243966766644196440 53235536608824717191588395651132620 63882643930589660629906074781359144 74149497419434427516855265371220435 83255576049891287036469069890493120 90432780881778466953406058674156320 95031057875767202561206366742333760 96614908840363322603893139521372656 95031057875767202561206366742333760 90432780881778466953406058674156320 83255576049891287036469069890493120 74149497419434427516855265371220435 63882643930589660629906074781359144 53235536608824717191588395651132620 42906253386216936243966766644196440 33441638668669082660738803413858990 25202394358996989831281417065516920 18361744461554949734219318147733756 12930805958841513897337547991361800 8800131833100474735688053494121225 5786388054641408045383925585175600 3675138359029002407203304087881800 2254084860204454809751359840567504 1334655509331585084721199905599180 762660291046620048412114231770960 420440929679546949765652717514760 223525557551151542913384989058480 114556848244965165743109806892471 56571283083933415181782620687640 26905854149675648684018563497780 12318342863706923493888017023080 5425936737585192491355436069690 2298043794741963878691714100104 935250381581031811095465040740 365500149123621627324664498680 137062555921358110246749187005 49280918982960219414561505440 16974538760797408909460074096 5596001789273871069052771680 1763957085749372402201417160 531083853774004594211179360 152545362254235362167040880 41749257038001257014137504 10872202353646160680764975 2690029448324823261220200 631333441953785051102700 140296320434174455800600 29462227291176635718126 5834104414094383310520 1086744939880326302940 189916591435396829640 31044058215401404845 4730523156632595024 669413654240461560 87586833265107120 10542859559688820 1160681786387760 116068178638776 10456592670160 840261910995 59487568920 3652745460 190578024 8214570 280840 7140 120 1);
                my @pascal_row121 =
                  qw(1 121 7260 287980 8495410 198792594 3843323484 63140314380 899749479915 11296854581155 126524771308936 1276749965026536 11703541346076580 98129692824795940 757000487505568680 5399936810873056584 35774581372033999869 220960649650798234485 1276661531315723132580 6920849353974709613460 35296331705271019028646 169758547725351091518726 771629762387959506903300 3321362890278608312322900 13562231801970983941985175 52621459391647417694902479 194294619292236619181178384 683629216028239956378220240 2295040939523376996412596520 7359958875023243471254188840 22570540550071279978512845776 66255457743757628324021579536 186343474904318329661310692445 502562705044979737571413685685 1300750530704653438420129539420 3233294176322995689787179140844 7723980532327156370047150169794 17744279601292115985243453092770 39224197013382572177906580520860 83477137233609063865801184185420 171128131328898580924892427580111 338082405796116708656494795950951 643966487230698492679037706573240 1183101220726166998177766949285720 2097315800378205133133314137370140 3588740369536039894472559746166684 5929223219233457216954663928449304 9461526413670410452587229673057400 14586519887741882781071979079296825 21730937791941988633025601485483025 31292550420396463631556866139095556 43564138820551939565500735213250676 58644033027666072492020220479375910 76347892054886018904705570058055430 96141789995041653435555162295329060 117118180539414377821494470432491764 138032141350024088146761340152579579 157405073469325714553324335261713555 173688356931669753989875128564649440 185463838757545669514612425416490080 191645966716130525165099506263706416 191645966716130525165099506263706416 185463838757545669514612425416490080 173688356931669753989875128564649440 157405073469325714553324335261713555 138032141350024088146761340152579579 117118180539414377821494470432491764 96141789995041653435555162295329060 76347892054886018904705570058055430 58644033027666072492020220479375910 43564138820551939565500735213250676 31292550420396463631556866139095556 21730937791941988633025601485483025 14586519887741882781071979079296825 9461526413670410452587229673057400 5929223219233457216954663928449304 3588740369536039894472559746166684 2097315800378205133133314137370140 1183101220726166998177766949285720 643966487230698492679037706573240 338082405796116708656494795950951 171128131328898580924892427580111 83477137233609063865801184185420 39224197013382572177906580520860 17744279601292115985243453092770 7723980532327156370047150169794 3233294176322995689787179140844 1300750530704653438420129539420 502562705044979737571413685685 186343474904318329661310692445 66255457743757628324021579536 22570540550071279978512845776 7359958875023243471254188840 2295040939523376996412596520 683629216028239956378220240 194294619292236619181178384 52621459391647417694902479 13562231801970983941985175 3321362890278608312322900 771629762387959506903300 169758547725351091518726 35296331705271019028646 6920849353974709613460 1276661531315723132580 220960649650798234485 35774581372033999869 5399936810873056584 757000487505568680 98129692824795940 11703541346076580 1276749965026536 126524771308936 11296854581155 899749479915 63140314380 3843323484 198792594 8495410 287980 7260 121 1);
                my @pascal_row122 =
                  qw(1 122 7381 295240 8783390 207288004 4042116078 66983637864 962889794295 12196604061070 137821625890091 1403274736335472 12980291311103116 109833234170872520 855130180330364620 6156937298378625264 41174518182907056453 256735231022832234354 1497622180966521367065 8197510885290432746040 42217181059245728642106 205054879430622110547372 941388310113310598422026 4092992652666567819226200 16883594692249592254308075 66183691193618401636887654 246916078683884036876080863 877923835320476575559398624 2978670155551616952790816760 9654999814546620467666785360 29930499425094523449767034616 88825998293828908302534425312 252598932648075957985332271981 688906179949298067232724378130 1803313235749633175991543225105 4534044707027649128207308680264 10957274708650152059834329310638 25468260133619272355290603262564 56968476614674688163150033613630 122701334246991636043707764706280 254605268562507644790693611765531 509210537125015289581387223531062 982048893026815201335532502524191 1827067707956865490856804655858960 3280417021104372131311081086655860 5686056169914245027605873883536824 9517963588769497111427223674615988 15390749632903867669541893601506704 24048046301412293233659208752354225 36317457679683871414097580564779850 53023488212338452264582467624578581 74856689240948403197057601352346232 102208171848218012057520955692626586 134991925082552091396725790537431340 172489682049927672340260732353384490 213259970534456031257049632727820824 255150321889438465968255810585071343 295437214819349802700085675414293134 331093430400995468543199463826362995 359152195689215423504487553981139520 377109805473676194679711931680196496 383291933432261050330199012527412832 377109805473676194679711931680196496 359152195689215423504487553981139520 331093430400995468543199463826362995 295437214819349802700085675414293134 255150321889438465968255810585071343 213259970534456031257049632727820824 172489682049927672340260732353384490 134991925082552091396725790537431340 102208171848218012057520955692626586 74856689240948403197057601352346232 53023488212338452264582467624578581 36317457679683871414097580564779850 24048046301412293233659208752354225 15390749632903867669541893601506704 9517963588769497111427223674615988 5686056169914245027605873883536824 3280417021104372131311081086655860 1827067707956865490856804655858960 982048893026815201335532502524191 509210537125015289581387223531062 254605268562507644790693611765531 122701334246991636043707764706280 56968476614674688163150033613630 25468260133619272355290603262564 10957274708650152059834329310638 4534044707027649128207308680264 1803313235749633175991543225105 688906179949298067232724378130 252598932648075957985332271981 88825998293828908302534425312 29930499425094523449767034616 9654999814546620467666785360 2978670155551616952790816760 877923835320476575559398624 246916078683884036876080863 66183691193618401636887654 16883594692249592254308075 4092992652666567819226200 941388310113310598422026 205054879430622110547372 42217181059245728642106 8197510885290432746040 1497622180966521367065 256735231022832234354 41174518182907056453 6156937298378625264 855130180330364620 109833234170872520 12980291311103116 1403274736335472 137821625890091 12196604061070 962889794295 66983637864 4042116078 207288004 8783390 295240 7381 122 1);
                my @pascal_row123 =
                  qw(1 123 7503 302621 9078630 216071394 4249404082 71025753942 1029873432159 13159493855365 150018229951161 1541096362225563 14383566047438588 122813525481975636 964963414501237140 7012067478708989884 47331455481285681717 297909749205739290807 1754357411989353601419 9695133066256954113105 50414691944536161388146 247272060489867839189478 1146443189543932708969398 5034380962779878417648226 20976587344916160073534275 83067285885867993891195729 313099769877502438512968517 1124839914004360612435479487 3856593990872093528350215384 12633669970098237420457602120 39585499239641143917433819976 118756497718923431752301459928 341424930941904866287866697293 941505112597374025218056650111 2492219415698931243224267603235 6337357942777282304198851905369 15491319415677801188041637990902 36425534842269424415124932573202 82436736748293960518440636876194 179669810861666324206857798319910 377306602809499280834401376471811 763815805687522934372080835296593 1491259430151830490916919726055253 2809116600983680692192337158383151 5107484729061237622167885742514820 8966473191018617158916954970192684 15204019758683742139033097558152812 24908713221673364780969117276122692 39438795934316160903201102353860929 60365503981096164647756789317134075 89340945892022323678680048189358431 127880177453286855461640068976924813 177064861089166415254578557044972818 237200096930770103454246746230057926 307481607132479763736986522890815830 385749652584383703597310365081205314 468410292423894497225305443312892167 550587536708788268668341485999364477 626530645220345271243285139240656129 690245626090210892047687017807502515 736262001162891618184199485661336016 760401738905937245009910944207609328 760401738905937245009910944207609328 736262001162891618184199485661336016 690245626090210892047687017807502515 626530645220345271243285139240656129 550587536708788268668341485999364477 468410292423894497225305443312892167 385749652584383703597310365081205314 307481607132479763736986522890815830 237200096930770103454246746230057926 177064861089166415254578557044972818 127880177453286855461640068976924813 89340945892022323678680048189358431 60365503981096164647756789317134075 39438795934316160903201102353860929 24908713221673364780969117276122692 15204019758683742139033097558152812 8966473191018617158916954970192684 5107484729061237622167885742514820 2809116600983680692192337158383151 1491259430151830490916919726055253 763815805687522934372080835296593 377306602809499280834401376471811 179669810861666324206857798319910 82436736748293960518440636876194 36425534842269424415124932573202 15491319415677801188041637990902 6337357942777282304198851905369 2492219415698931243224267603235 941505112597374025218056650111 341424930941904866287866697293 118756497718923431752301459928 39585499239641143917433819976 12633669970098237420457602120 3856593990872093528350215384 1124839914004360612435479487 313099769877502438512968517 83067285885867993891195729 20976587344916160073534275 5034380962779878417648226 1146443189543932708969398 247272060489867839189478 50414691944536161388146 9695133066256954113105 1754357411989353601419 297909749205739290807 47331455481285681717 7012067478708989884 964963414501237140 122813525481975636 14383566047438588 1541096362225563 150018229951161 13159493855365 1029873432159 71025753942 4249404082 216071394 9078630 302621 7503 123 1);
                my @pascal_row124 =
                  qw(1 124 7626 310124 9381251 225150024 4465475476 75275158024 1100899186101 14189367287524 163177723806526 1691114592176724 15924662409664151 137197091529414224 1087776939983212776 7977030893210227024 54343522959994671601 345241204687024972524 2052267161195092892226 11449490478246307714524 60109825010793115501251 297686752434404000577624 1393715250033800548158876 6180824152323811126617624 26010968307696038491182501 104043873230784153964730004 396167055763370432404164246 1437939683881863050948448004 4981433904876454140785694871 16490263960970330948807817504 52219169209739381337891422096 158341996958564575669735279904 460181428660828298040168157221 1282930043539278891505923347404 3433724528296305268442324253346 8829577358476213547423119508604 21828677358455083492240489896271 51916854257947225603166570564104 118862271590563384933565569449396 262106547609960284725298435196104 556976413671165605041259174791721 1141122408497022215206482211768404 2255075235839353425289000561351846 4300376031135511183109256884438404 7916601330044918314360222900897971 14073957920079854781084840712707504 24170492949702359297950052528345496 40112732980357106920002214834275504 64347509155989525684170219629983621 99804299915412325550957891670995004 149706449873118488326436837506492506 217221123345309179140320117166283244 304945038542453270716218626021897631 414264958019936518708825303275030744 544681704063249867191233269120873756 693231259716863467334296887972021144 854159945008278200822615808394097481 1018997829132682765893646929312256644 1177118181929133539911626625240020606 1316776271310556163290972157048158644 1426507627253102510231886503468838531 1496663740068828863194110429868945344 1520803477811874490019821888415218656 1496663740068828863194110429868945344 1426507627253102510231886503468838531 1316776271310556163290972157048158644 1177118181929133539911626625240020606 1018997829132682765893646929312256644 854159945008278200822615808394097481 693231259716863467334296887972021144 544681704063249867191233269120873756 414264958019936518708825303275030744 304945038542453270716218626021897631 217221123345309179140320117166283244 149706449873118488326436837506492506 99804299915412325550957891670995004 64347509155989525684170219629983621 40112732980357106920002214834275504 24170492949702359297950052528345496 14073957920079854781084840712707504 7916601330044918314360222900897971 4300376031135511183109256884438404 2255075235839353425289000561351846 1141122408497022215206482211768404 556976413671165605041259174791721 262106547609960284725298435196104 118862271590563384933565569449396 51916854257947225603166570564104 21828677358455083492240489896271 8829577358476213547423119508604 3433724528296305268442324253346 1282930043539278891505923347404 460181428660828298040168157221 158341996958564575669735279904 52219169209739381337891422096 16490263960970330948807817504 4981433904876454140785694871 1437939683881863050948448004 396167055763370432404164246 104043873230784153964730004 26010968307696038491182501 6180824152323811126617624 1393715250033800548158876 297686752434404000577624 60109825010793115501251 11449490478246307714524 2052267161195092892226 345241204687024972524 54343522959994671601 7977030893210227024 1087776939983212776 137197091529414224 15924662409664151 1691114592176724 163177723806526 14189367287524 1100899186101 75275158024 4465475476 225150024 9381251 310124 7626 124 1);
                my @pascal_row125 =
                  qw(1 125 7750 317750 9691375 234531275 4690625500 79740633500 1176174344125 15290266473625 177367091094050 1854292315983250 17615777001840875 153121753939078375 1224974031512627000 9064807833193439800 62320553853204898625 399584727647019644125 2397508365882117864750 13501757639441400606750 71559315489039423215775 357796577445197116078875 1691402002468204548736500 7574539402357611674776500 32191792460019849617800125 130054841538480192455912505 500210928994154586368894250 1834106739645233483352612250 6419373588758317191734142875 21471697865846785089593512375 68709433170709712286699239600 210561166168303957007626702000 618523425619392873709903437125 1743111472200107189546091504625 4716654571835584159948247600750 12263301886772518815865443761950 30658254716931297039663609404875 73745531616402309095407060460375 170779125848510610536732140013500 380968819200523669658864004645500 819082961281125889766557609987825 1698098822168187820247741386560125 3396197644336375640495482773120250 6555451266974864608398257445790250 12216977361180429497469479785336375 21990559250124773095445063613605475 38244450869782214079034893241053000 64283225930059466217952267362621000 104460242136346632604172434464259125 164151809071401851235128111300978625 249510749788530813877394729177487510 366927573218427667466756954672775750 522166161887762449856538743188180875 719209996562389789425043929296928375 958946662083186385900058572395904500 1237912963780113334525530157092894900 1547391204725141668156912696366118625 1873157774140960966716262737706354125 2196116011061816305805273554552277250 2493894453239689703202598782288179250 2743283898563658673522858660516997175 2923171367321931373425996933337783875 3017467217880703353213932318284164000 3017467217880703353213932318284164000 2923171367321931373425996933337783875 2743283898563658673522858660516997175 2493894453239689703202598782288179250 2196116011061816305805273554552277250 1873157774140960966716262737706354125 1547391204725141668156912696366118625 1237912963780113334525530157092894900 958946662083186385900058572395904500 719209996562389789425043929296928375 522166161887762449856538743188180875 366927573218427667466756954672775750 249510749788530813877394729177487510 164151809071401851235128111300978625 104460242136346632604172434464259125 64283225930059466217952267362621000 38244450869782214079034893241053000 21990559250124773095445063613605475 12216977361180429497469479785336375 6555451266974864608398257445790250 3396197644336375640495482773120250 1698098822168187820247741386560125 819082961281125889766557609987825 380968819200523669658864004645500 170779125848510610536732140013500 73745531616402309095407060460375 30658254716931297039663609404875 12263301886772518815865443761950 4716654571835584159948247600750 1743111472200107189546091504625 618523425619392873709903437125 210561166168303957007626702000 68709433170709712286699239600 21471697865846785089593512375 6419373588758317191734142875 1834106739645233483352612250 500210928994154586368894250 130054841538480192455912505 32191792460019849617800125 7574539402357611674776500 1691402002468204548736500 357796577445197116078875 71559315489039423215775 13501757639441400606750 2397508365882117864750 399584727647019644125 62320553853204898625 9064807833193439800 1224974031512627000 153121753939078375 17615777001840875 1854292315983250 177367091094050 15290266473625 1176174344125 79740633500 4690625500 234531275 9691375 317750 7750 125 1);
                my @pascal_row126 =
                  qw(1 126 7875 325500 10009125 244222650 4925156775 84431259000 1255914977625 16466440817750 192657357567675 2031659407077300 19470069317824125 170737530940919250 1378095785451705375 10289781864706066800 71385361686398338425 461905281500224542750 2797093093529137508875 15899266005323518471500 85061073128480823822525 429355892934236539294650 2049198579913401664815375 9265941404825816223513000 39766331862377461292576625 162246633998500042073712630 630265770532634778824806755 2334317668639388069721506500 8253480328403550675086755125 27891071454605102281327655250 90181131036556497376292751975 279270599339013669294325941600 829084591787696830717530139125 2361634897819500063255994941750 6459766044035691349494339105375 16979956458608102975813691362700 42921556603703815855529053166825 104403786333333606135070669865250 244524657464912919632139200473875 551747945049034280195596144659000 1200051780481649559425421614633325 2517181783449313710014298996547950 5094296466504563460743224159680375 9951648911311240248893740218910500 18772428628155294105867737231126625 34207536611305202592914543398941850 60235010119906987174479956854658475 102527676799841680296987160603674000 168743468066406098822124701826880125 268612051207748483839300545765237750 413662558859932665112522840478466135 616438323006958481344151683850263260 889093735106190117323295697860956625 1241376158450152239281582672485109250 1678156658645576175325102501692832875 2196859625863299720425588729488799400 2785304168505255002682442853459013525 3420548978866102634873175434072472750 4069273785202777272521536292258631375 4690010464301506009007872336840456500 5237178351803348376725457442805176425 5666455265885590046948855593854781050 5940638585202634726639929251621947875 6034934435761406706427864636568328000 5940638585202634726639929251621947875 5666455265885590046948855593854781050 5237178351803348376725457442805176425 4690010464301506009007872336840456500 4069273785202777272521536292258631375 3420548978866102634873175434072472750 2785304168505255002682442853459013525 2196859625863299720425588729488799400 1678156658645576175325102501692832875 1241376158450152239281582672485109250 889093735106190117323295697860956625 616438323006958481344151683850263260 413662558859932665112522840478466135 268612051207748483839300545765237750 168743468066406098822124701826880125 102527676799841680296987160603674000 60235010119906987174479956854658475 34207536611305202592914543398941850 18772428628155294105867737231126625 9951648911311240248893740218910500 5094296466504563460743224159680375 2517181783449313710014298996547950 1200051780481649559425421614633325 551747945049034280195596144659000 244524657464912919632139200473875 104403786333333606135070669865250 42921556603703815855529053166825 16979956458608102975813691362700 6459766044035691349494339105375 2361634897819500063255994941750 829084591787696830717530139125 279270599339013669294325941600 90181131036556497376292751975 27891071454605102281327655250 8253480328403550675086755125 2334317668639388069721506500 630265770532634778824806755 162246633998500042073712630 39766331862377461292576625 9265941404825816223513000 2049198579913401664815375 429355892934236539294650 85061073128480823822525 15899266005323518471500 2797093093529137508875 461905281500224542750 71385361686398338425 10289781864706066800 1378095785451705375 170737530940919250 19470069317824125 2031659407077300 192657357567675 16466440817750 1255914977625 84431259000 4925156775 244222650 10009125 325500 7875 126 1);
                my @pascal_row127 =
                  qw(1 127 8001 333375 10334625 254231775 5169379425 89356415775 1340346236625 17722355795375 209123798385425 2224316764644975 21501728724901425 190207600258743375 1548833316392624625 11667877650157772175 81675143551104405225 533290643186622881175 3258998375029362051625 18696359098852655980375 100960339133804342294025 514416966062717363117175 2478554472847638204110025 11315139984739217888328375 49032273267203277516089625 202012965860877503366289255 792512404531134820898519385 2964583439172022848546313255 10587797997042938744808261625 36144551783008652956414410375 118072202491161599657620407225 369451730375570166670618693575 1108355191126710500011856080725 3190719489607196893973525080875 8821400941855191412750334047125 23439722502643794325308030468075 59901513062311918831342744529525 147325342937037421990599723032075 348928443798246525767209870339125 796272602513947199827735345132875 1751799725530683839621017759292325 3717233563930963269439720611181275 7611478249953877170757523156228325 15045945377815803709636964378590875 28724077539466534354761477450037125 52979965239460496698782280630068475 94442546731212189767394500253600325 162762686919748667471467117458332475 271271144866247779119111862430554125 437355519274154582661425247592117875 682274610067681148951823386243703885 1030100881866891146456674524328729395 1505532058113148598667447381711219885 2130469893556342356604878370346065875 2919532817095728414606685174177942125 3875016284508875895750691231181632275 4982163794368554723108031582947812925 6205853147371357637555618287531486275 7489822764068879907394711726331104125 8759284249504283281529408629099087875 9927188816104854385733329779645632925 10903633617688938423674313036659957475 11607093851088224773588784845476728925 11975573020964041433067793888190275875 11975573020964041433067793888190275875 11607093851088224773588784845476728925 10903633617688938423674313036659957475 9927188816104854385733329779645632925 8759284249504283281529408629099087875 7489822764068879907394711726331104125 6205853147371357637555618287531486275 4982163794368554723108031582947812925 3875016284508875895750691231181632275 2919532817095728414606685174177942125 2130469893556342356604878370346065875 1505532058113148598667447381711219885 1030100881866891146456674524328729395 682274610067681148951823386243703885 437355519274154582661425247592117875 271271144866247779119111862430554125 162762686919748667471467117458332475 94442546731212189767394500253600325 52979965239460496698782280630068475 28724077539466534354761477450037125 15045945377815803709636964378590875 7611478249953877170757523156228325 3717233563930963269439720611181275 1751799725530683839621017759292325 796272602513947199827735345132875 348928443798246525767209870339125 147325342937037421990599723032075 59901513062311918831342744529525 23439722502643794325308030468075 8821400941855191412750334047125 3190719489607196893973525080875 1108355191126710500011856080725 369451730375570166670618693575 118072202491161599657620407225 36144551783008652956414410375 10587797997042938744808261625 2964583439172022848546313255 792512404531134820898519385 202012965860877503366289255 49032273267203277516089625 11315139984739217888328375 2478554472847638204110025 514416966062717363117175 100960339133804342294025 18696359098852655980375 3258998375029362051625 533290643186622881175 81675143551104405225 11667877650157772175 1548833316392624625 190207600258743375 21501728724901425 2224316764644975 209123798385425 17722355795375 1340346236625 89356415775 5169379425 254231775 10334625 333375 8001 127 1);
                my @pascal_row128 =
                  qw(1 128 8128 341376 10668000 264566400 5423611200 94525795200 1429702652400 19062702032000 226846154180800 2433440563030400 23726045489546400 211709328983644800 1739040916651368000 13216710966550396800 93343021201262177400 614965786737727286400 3792289018215984932800 21955357473882018032000 119656698232656998274400 615377305196521705411200 2992971438910355567227200 13793694457586856092438400 60347413251942495404418000 251045239128080780882378880 994525370392012324264808640 3757095843703157669444832640 13552381436214961593354574880 46732349780051591701222672000 154216754274170252614034817600 487523932866731766328239100800 1477806921502280666682474774300 4299074680733907393985381161600 12012120431462388306723859128000 32261123444498985738058364515200 83341235564955713156650774997600 207226855999349340821942467561600 496253786735283947757809593371200 1145201046312193725594945215472000 2548072328044631039448753104425200 5469033289461647109060738370473600 11328711813884840440197243767409600 22657423627769680880394487534819200 43770022917282338064398441828628000 81704042778927031053543758080105600 147422511970672686466176780883668800 257205233650960857238861617711932800 434033831785996446590578979888886600 708626664140402361780537110022672000 1119630129341835731613248633835821760 1712375491934572295408497910572433280 2535632939980039745124121906039949280 3636001951669490955272325752057285760 5050002710652070771211563544524008000 6794549101604604310357376405359574400 8857180078877430618858722814129445200 11188016941739912360663649870479299200 13695675911440237544950330013862590400 16249107013573163188924120355430192000 18686473065609137667262738408744720800 20830822433793792809407642816305590400 22510727468777163197263097882136686400 23582666872052266206656578733667004800 23951146041928082866135587776380551750 23582666872052266206656578733667004800 22510727468777163197263097882136686400 20830822433793792809407642816305590400 18686473065609137667262738408744720800 16249107013573163188924120355430192000 13695675911440237544950330013862590400 11188016941739912360663649870479299200 8857180078877430618858722814129445200 6794549101604604310357376405359574400 5050002710652070771211563544524008000 3636001951669490955272325752057285760 2535632939980039745124121906039949280 1712375491934572295408497910572433280 1119630129341835731613248633835821760 708626664140402361780537110022672000 434033831785996446590578979888886600 257205233650960857238861617711932800 147422511970672686466176780883668800 81704042778927031053543758080105600 43770022917282338064398441828628000 22657423627769680880394487534819200 11328711813884840440197243767409600 5469033289461647109060738370473600 2548072328044631039448753104425200 1145201046312193725594945215472000 496253786735283947757809593371200 207226855999349340821942467561600 83341235564955713156650774997600 32261123444498985738058364515200 12012120431462388306723859128000 4299074680733907393985381161600 1477806921502280666682474774300 487523932866731766328239100800 154216754274170252614034817600 46732349780051591701222672000 13552381436214961593354574880 3757095843703157669444832640 994525370392012324264808640 251045239128080780882378880 60347413251942495404418000 13793694457586856092438400 2992971438910355567227200 615377305196521705411200 119656698232656998274400 21955357473882018032000 3792289018215984932800 614965786737727286400 93343021201262177400 13216710966550396800 1739040916651368000 211709328983644800 23726045489546400 2433440563030400 226846154180800 19062702032000 1429702652400 94525795200 5423611200 264566400 10668000 341376 8128 128 1);
                my @pascal_row129 =
                  qw(1 129 8256 349504 11009376 275234400 5688177600 99949406400 1524228447600 20492404684400 245908856212800 2660286717211200 26159486052576800 235435374473191200 1950750245635012800 14955751883201764800 106559732167812574200 708308807938989463800 4407254804953712219200 25747646492098002964800 141612055706539016306400 735034003429178703685600 3608348744106877272638400 16786665896497211659665600 74141107709529351496856400 311392652380023276286796880 1245570609520093105147187520 4751621214095169993709641280 17309477279918119262799407520 60284731216266553294577246880 200949104054221844315257489600 641740687140902018942273918400 1965330854369012433010713875100 5776881602236188060667855935900 16311195112196295700709240289600 44273243875961374044782223643200 115602359009454698894709139512800 290568091564305053978593242559200 703480642734633288579752060932800 1641454833047477673352754808843200 3693273374356824765043698319897200 8017105617506278148509491474898800 16797745103346487549257982137883200 33986135441654521320591731302228800 66427446545052018944792929363447200 125474065696209369117942199908733600 229126554749599717519720538963774400 404627745621633543705038398595601600 691239065436957303829440597600819400 1142660495926398808371116089911558600 1828256793482238093393785743858493760 2832005621276408027021746544408255040 4248008431914612040532619816612382560 6171634891649530700396447658097235040 8686004662321561726483889296581293760 11844551812256675081568939949883582400 15651729180482034929216099219489019600 20045197020617342979522372684608744400 24883692853180149905613979884341889600 29944782925013400733874450369292782400 34935580079182300856186858764174912800 39517295499402930476670381225050311200 43341549902570956006670740698442276800 46093394340829429403919676615803691200 47533812913980349072792166510047556550 47533812913980349072792166510047556550 46093394340829429403919676615803691200 43341549902570956006670740698442276800 39517295499402930476670381225050311200 34935580079182300856186858764174912800 29944782925013400733874450369292782400 24883692853180149905613979884341889600 20045197020617342979522372684608744400 15651729180482034929216099219489019600 11844551812256675081568939949883582400 8686004662321561726483889296581293760 6171634891649530700396447658097235040 4248008431914612040532619816612382560 2832005621276408027021746544408255040 1828256793482238093393785743858493760 1142660495926398808371116089911558600 691239065436957303829440597600819400 404627745621633543705038398595601600 229126554749599717519720538963774400 125474065696209369117942199908733600 66427446545052018944792929363447200 33986135441654521320591731302228800 16797745103346487549257982137883200 8017105617506278148509491474898800 3693273374356824765043698319897200 1641454833047477673352754808843200 703480642734633288579752060932800 290568091564305053978593242559200 115602359009454698894709139512800 44273243875961374044782223643200 16311195112196295700709240289600 5776881602236188060667855935900 1965330854369012433010713875100 641740687140902018942273918400 200949104054221844315257489600 60284731216266553294577246880 17309477279918119262799407520 4751621214095169993709641280 1245570609520093105147187520 311392652380023276286796880 74141107709529351496856400 16786665896497211659665600 3608348744106877272638400 735034003429178703685600 141612055706539016306400 25747646492098002964800 4407254804953712219200 708308807938989463800 106559732167812574200 14955751883201764800 1950750245635012800 235435374473191200 26159486052576800 2660286717211200 245908856212800 20492404684400 1524228447600 99949406400 5688177600 275234400 11009376 349504 8256 129 1);
                my @pascal_row130 =
                  qw(1 130 8385 357760 11358880 286243776 5963412000 105637584000 1624177854000 22016633132000 266401260897200 2906195573424000 28819772769788000 261594860525768000 2186185620108204000 16906502128836777600 121515484051014339000 814868540106802038000 5115563612892701683000 30154901297051715184000 167359702198637019271200 876646059135717719992000 4343382747536055976324000 20395014640604088932304000 90927773606026563156522000 385533760089552627783653280 1556963261900116381433984400 5997191823615263098856828800 22061098494013289256509048800 77594208496184672557376654400 261233835270488397609834736480 842689791195123863257531408000 2607071541509914451952987793500 7742212456605200493678569811000 22088076714432483761377096225500 60584438988157669745491463932800 159875602885416072939491363156000 406170450573759752873302382072000 994048734298938342558345303492000 2344935475782110961932506869776000 5334728207404302438396453128740400 11710378991863102913553189794796000 24814850720852765697767473612782000 50783880545001008869849713440112000 100413581986706540265384660665676000 191901512241261388062735129272180800 354600620445809086637662738872508000 633754300371233261224758937559376000 1095866811058590847534478996196421000 1833899561363356112200556687512378000 2970917289408636901764901833770052360 4660262414758646120415532288266748800 7080014053191020067554366361020637600 10419643323564142740929067474709617600 14857639553971092426880336954678528800 20530556474578236808052829246464876160 27496280992738710010785039169372602000 35696926201099377908738471904097764000 44928889873797492885136352568950634000 54828475778193550639488430253634672000 64880363004195701590061309133467695200 74452875578585231332857239989225224000 82858845401973886483341121923492588000 89434944243400385410590417314245968000 93627207254809778476711843125851247750 95067625827960698145584333020095113100 93627207254809778476711843125851247750 89434944243400385410590417314245968000 82858845401973886483341121923492588000 74452875578585231332857239989225224000 64880363004195701590061309133467695200 54828475778193550639488430253634672000 44928889873797492885136352568950634000 35696926201099377908738471904097764000 27496280992738710010785039169372602000 20530556474578236808052829246464876160 14857639553971092426880336954678528800 10419643323564142740929067474709617600 7080014053191020067554366361020637600 4660262414758646120415532288266748800 2970917289408636901764901833770052360 1833899561363356112200556687512378000 1095866811058590847534478996196421000 633754300371233261224758937559376000 354600620445809086637662738872508000 191901512241261388062735129272180800 100413581986706540265384660665676000 50783880545001008869849713440112000 24814850720852765697767473612782000 11710378991863102913553189794796000 5334728207404302438396453128740400 2344935475782110961932506869776000 994048734298938342558345303492000 406170450573759752873302382072000 159875602885416072939491363156000 60584438988157669745491463932800 22088076714432483761377096225500 7742212456605200493678569811000 2607071541509914451952987793500 842689791195123863257531408000 261233835270488397609834736480 77594208496184672557376654400 22061098494013289256509048800 5997191823615263098856828800 1556963261900116381433984400 385533760089552627783653280 90927773606026563156522000 20395014640604088932304000 4343382747536055976324000 876646059135717719992000 167359702198637019271200 30154901297051715184000 5115563612892701683000 814868540106802038000 121515484051014339000 16906502128836777600 2186185620108204000 261594860525768000 28819772769788000 2906195573424000 266401260897200 22016633132000 1624177854000 105637584000 5963412000 286243776 11358880 357760 8385 130 1);
                my @pascal_row131 =
                  qw(1 131 8515 366145 11716640 297602656 6249655776 111600996000 1729815438000 23640810986000 288417894029200 3172596834321200 31725968343212000 290414633295556000 2447780480633972000 19092687748944981600 138421986179851116600 936384024157816377000 5930432152999503721000 35270464909944416867000 197514603495688734455200 1044005761334354739263200 5220028806671773696316000 24738397388140144908628000 111322788246630652088826000 476461533695579190940175280 1942497021989669009217637680 7554155085515379480290813200 28058290317628552355365877600 99655306990197961813885703200 338828043766673070167211390880 1103923626465612260867366144480 3449761332705038315210519201500 10349283998115114945631557604500 29830289171037684255055666036500 82672515702590153506868560158300 220460041873573742684982827088800 566046053459175825812793745228000 1400219184872698095431647685564000 3338984210081049304490852173268000 7679663683186413400328959998516400 17045107199267405351949642923536400 36525229712715868611320663407578000 75598731265853774567617187052894000 151197462531707549135234374105788000 292315094227967928328119789937856800 546502132687070474700397868144688800 988354920817042347862421676431884000 1729621111429824108759237933755797000 2929766372421946959735035683708799000 4804816850771993013965458521282430360 7631179704167283022180434122036801160 11740276467949666187969898649287386400 17499657376755162808483433835730255200 25277282877535235167809404429388146400 35388196028549329234933166201143404960 48026837467316946818837868415837478160 63193207193838087919523511073470366000 80625816074896870793874824473048398000 99757365651991043524624782822585306000 119708838782389252229549739387102367200 139333238582780932922918549122692919200 157311720980559117816198361912717812000 172293789645374271893931539237738556000 183062151498210163887302260440097215750 188694833082770476622296176145946360850 188694833082770476622296176145946360850 183062151498210163887302260440097215750 172293789645374271893931539237738556000 157311720980559117816198361912717812000 139333238582780932922918549122692919200 119708838782389252229549739387102367200 99757365651991043524624782822585306000 80625816074896870793874824473048398000 63193207193838087919523511073470366000 48026837467316946818837868415837478160 35388196028549329234933166201143404960 25277282877535235167809404429388146400 17499657376755162808483433835730255200 11740276467949666187969898649287386400 7631179704167283022180434122036801160 4804816850771993013965458521282430360 2929766372421946959735035683708799000 1729621111429824108759237933755797000 988354920817042347862421676431884000 546502132687070474700397868144688800 292315094227967928328119789937856800 151197462531707549135234374105788000 75598731265853774567617187052894000 36525229712715868611320663407578000 17045107199267405351949642923536400 7679663683186413400328959998516400 3338984210081049304490852173268000 1400219184872698095431647685564000 566046053459175825812793745228000 220460041873573742684982827088800 82672515702590153506868560158300 29830289171037684255055666036500 10349283998115114945631557604500 3449761332705038315210519201500 1103923626465612260867366144480 338828043766673070167211390880 99655306990197961813885703200 28058290317628552355365877600 7554155085515379480290813200 1942497021989669009217637680 476461533695579190940175280 111322788246630652088826000 24738397388140144908628000 5220028806671773696316000 1044005761334354739263200 197514603495688734455200 35270464909944416867000 5930432152999503721000 936384024157816377000 138421986179851116600 19092687748944981600 2447780480633972000 290414633295556000 31725968343212000 3172596834321200 288417894029200 23640810986000 1729815438000 111600996000 6249655776 297602656 11716640 366145 8515 131 1);
                my @pascal_row132 =
                  qw(1 132 8646 374660 12082785 309319296 6547258432 117850651776 1841416434000 25370626424000 312058705015200 3461014728350400 34898565177533200 322140601638768000 2738195113929528000 21540468229578953600 157514673928796098200 1074806010337667493600 6866816177157320098000 41200897062943920588000 232785068405633151322200 1241520364830043473718400 6264034568006128435579200 29958426194811918604944000 136061185634770796997454000 587784321942209843029001280 2418958555685248200157812960 9496652107505048489508450880 35612445403143931835656690800 127713597307826514169251580800 438483350756871031981097094080 1442751670232285331034577535360 4553684959170650576077885345980 13799045330820153260842076806000 40179573169152799200687223641000 112502804873627837761924226194800 303132557576163896191851387247100 786506095332749568497776572316800 1966265238331873921244441430792000 4739203394953747399922499858832000 11018647893267462704819812171784400 24724770882453818752278602922052800 53570336911983273963270306331114400 112123960978569643178937850460472000 226796193797561323702851561158682000 443512556759675477463354164043644800 838817226915038403028517658082545600 1534857053504112822562819544576572800 2717976032246866456621659610187681000 4659387483851771068494273617464596000 7734583223193939973700494204991229360 12435996554939276036145892643319231520 19371456172116949210150332771324187560 29239933844704828996453332485017641600 42776940254290397976292838265118401600 60665478906084564402742570630531551360 83415033495866276053771034616980883120 111220044661155034738361379489307844160 143819023268734958713398335546518764000 180383181726887914318499607295633704000 219466204434380295754174522209687673200 259042077365170185152468288509795286400 296644959563340050739116911035410731200 329605510625933389710129901150456368000 355355941143584435781233799677835771750 371756984580980640509598436586043576600 377389666165540953244592352291892721700 371756984580980640509598436586043576600 355355941143584435781233799677835771750 329605510625933389710129901150456368000 296644959563340050739116911035410731200 259042077365170185152468288509795286400 219466204434380295754174522209687673200 180383181726887914318499607295633704000 143819023268734958713398335546518764000 111220044661155034738361379489307844160 83415033495866276053771034616980883120 60665478906084564402742570630531551360 42776940254290397976292838265118401600 29239933844704828996453332485017641600 19371456172116949210150332771324187560 12435996554939276036145892643319231520 7734583223193939973700494204991229360 4659387483851771068494273617464596000 2717976032246866456621659610187681000 1534857053504112822562819544576572800 838817226915038403028517658082545600 443512556759675477463354164043644800 226796193797561323702851561158682000 112123960978569643178937850460472000 53570336911983273963270306331114400 24724770882453818752278602922052800 11018647893267462704819812171784400 4739203394953747399922499858832000 1966265238331873921244441430792000 786506095332749568497776572316800 303132557576163896191851387247100 112502804873627837761924226194800 40179573169152799200687223641000 13799045330820153260842076806000 4553684959170650576077885345980 1442751670232285331034577535360 438483350756871031981097094080 127713597307826514169251580800 35612445403143931835656690800 9496652107505048489508450880 2418958555685248200157812960 587784321942209843029001280 136061185634770796997454000 29958426194811918604944000 6264034568006128435579200 1241520364830043473718400 232785068405633151322200 41200897062943920588000 6866816177157320098000 1074806010337667493600 157514673928796098200 21540468229578953600 2738195113929528000 322140601638768000 34898565177533200 3461014728350400 312058705015200 25370626424000 1841416434000 117850651776 6547258432 309319296 12082785 374660 8646 132 1);
                my @pascal_row133 =
                  qw(1 133 8778 383306 12457445 321402081 6856577728 124397910208 1959267085776 27212042858000 337429331439200 3773073433365600 38359579905883600 357039166816301200 3060335715568296000 24278663343508481600 179055142158375051800 1232320684266463591800 7941622187494987591600 48067713240101240686000 273985965468577071910200 1474305433235676625040600 7505554932836171909297600 36222460762818047040523200 166019611829582715602398000 723845507576980640026455280 3006742877627458043186814240 11915610663190296689666263840 45109097510648980325165141680 163326042710970446004908271600 566196948064697546150348674880 1881235020989156363015674629440 5996436629402935907112462881340 18352730289990803836919962151980 53978618499972952461529300447000 152682378042780636962611449835800 415635362449791733953775613441900 1089638652908913464689627959563900 2752771333664623489742218003108800 6705468633285621321166941289624000 15757851288221210104742312030616400 35743418775721281457098415093837200 78295107794437092715548909253167200 165694297890552917142208156791586400 338920154776130966881789411619154000 670308750557236801166205725202326800 1282329783674713880491871822126190400 2373674280419151225591337202659118400 4252833085750979279184479154764253800 7377363516098637525115933227652277000 12393970707045711042194767822455825360 20170579778133216009846386848310460880 31807452727056225246296225414643419080 48611390016821778206603665256341829160 72016874098995226972746170750136043200 103442419160374962379035408895649952960 144080512401950840456513605247512434480 194635078157021310792132414106288727280 255039067929889993451759715035826608160 324202204995622873031897942842152468000 399849386161268210072674129505321377200 478508281799550480906642810719482959600 555687036928510235891585199545206017600 626250470189273440449246812185867099200 684961451769517825491363700828292139750 727112925724565076290832236263879348350 749146650746521593754190788877936298300 749146650746521593754190788877936298300 727112925724565076290832236263879348350 684961451769517825491363700828292139750 626250470189273440449246812185867099200 555687036928510235891585199545206017600 478508281799550480906642810719482959600 399849386161268210072674129505321377200 324202204995622873031897942842152468000 255039067929889993451759715035826608160 194635078157021310792132414106288727280 144080512401950840456513605247512434480 103442419160374962379035408895649952960 72016874098995226972746170750136043200 48611390016821778206603665256341829160 31807452727056225246296225414643419080 20170579778133216009846386848310460880 12393970707045711042194767822455825360 7377363516098637525115933227652277000 4252833085750979279184479154764253800 2373674280419151225591337202659118400 1282329783674713880491871822126190400 670308750557236801166205725202326800 338920154776130966881789411619154000 165694297890552917142208156791586400 78295107794437092715548909253167200 35743418775721281457098415093837200 15757851288221210104742312030616400 6705468633285621321166941289624000 2752771333664623489742218003108800 1089638652908913464689627959563900 415635362449791733953775613441900 152682378042780636962611449835800 53978618499972952461529300447000 18352730289990803836919962151980 5996436629402935907112462881340 1881235020989156363015674629440 566196948064697546150348674880 163326042710970446004908271600 45109097510648980325165141680 11915610663190296689666263840 3006742877627458043186814240 723845507576980640026455280 166019611829582715602398000 36222460762818047040523200 7505554932836171909297600 1474305433235676625040600 273985965468577071910200 48067713240101240686000 7941622187494987591600 1232320684266463591800 179055142158375051800 24278663343508481600 3060335715568296000 357039166816301200 38359579905883600 3773073433365600 337429331439200 27212042858000 1959267085776 124397910208 6856577728 321402081 12457445 383306 8778 133 1);
                my @pascal_row134 =
                  qw(1 134 8911 392084 12840751 333859526 7177979809 131254487936 2083664995984 29171309943776 364641374297200 4110502764804800 42132653339249200 395398746722184800 3417374882384597200 27338999059076777600 203333805501883533400 1411375826424838643600 9173942871761451183400 56009335427596228277600 322053678708678312596200 1748291398704253696950800 8979860366071848534338200 43728015695654218949820800 202242072592400762642921200 889865119406563355628853280 3730588385204438683213269520 14922353540817754732853078080 57024708173839277014831405520 208435140221619426330073413280 729522990775667992155256946480 2447431969053853909166023304320 7877671650392092270128137510780 24349166919393739744032425033320 72331348789963756298449262598980 206660996542753589424140750282800 568317740492572370916387063277700 1505274015358705198643403573005800 3842409986573536954431845962672700 9458239966950244810909159292732800 22463319921506831425909253320240400 51501270063942491561840727124453600 114038526570158374172647324347004400 243989405684990009857757066044753600 504614452666683884023997568410740400 1009228905333367768047995136821480800 1952638534231950681658077547328517200 3656004064093865106083209024785308800 6626507366170130504775816357423372200 11630196601849616804300412382416530800 19771334223144348567310701050108102360 32564550485178927052041154670766286240 51978032505189441256142612262953879960 80418842743878003452899890670985248240 120628264115817005179349836006477872360 175459293259370189351781579645785996160 247522931562325802835549014143162387440 338715590558972151248646019353801161760 449674146086911304243892129142115335440 579241272925512866483657657877979076160 724051591156891083104572072347473845200 878357667960818690979316940224804336800 1034195318728060716798228010264688977200 1181937507117783676340832011731073116800 1311211921958791265940610513014159238950 1412074377494082901782195937092171488100 1476259576471086670045023025141815646650 1498293301493043187508381577755872596600 1476259576471086670045023025141815646650 1412074377494082901782195937092171488100 1311211921958791265940610513014159238950 1181937507117783676340832011731073116800 1034195318728060716798228010264688977200 878357667960818690979316940224804336800 724051591156891083104572072347473845200 579241272925512866483657657877979076160 449674146086911304243892129142115335440 338715590558972151248646019353801161760 247522931562325802835549014143162387440 175459293259370189351781579645785996160 120628264115817005179349836006477872360 80418842743878003452899890670985248240 51978032505189441256142612262953879960 32564550485178927052041154670766286240 19771334223144348567310701050108102360 11630196601849616804300412382416530800 6626507366170130504775816357423372200 3656004064093865106083209024785308800 1952638534231950681658077547328517200 1009228905333367768047995136821480800 504614452666683884023997568410740400 243989405684990009857757066044753600 114038526570158374172647324347004400 51501270063942491561840727124453600 22463319921506831425909253320240400 9458239966950244810909159292732800 3842409986573536954431845962672700 1505274015358705198643403573005800 568317740492572370916387063277700 206660996542753589424140750282800 72331348789963756298449262598980 24349166919393739744032425033320 7877671650392092270128137510780 2447431969053853909166023304320 729522990775667992155256946480 208435140221619426330073413280 57024708173839277014831405520 14922353540817754732853078080 3730588385204438683213269520 889865119406563355628853280 202242072592400762642921200 43728015695654218949820800 8979860366071848534338200 1748291398704253696950800 322053678708678312596200 56009335427596228277600 9173942871761451183400 1411375826424838643600 203333805501883533400 27338999059076777600 3417374882384597200 395398746722184800 42132653339249200 4110502764804800 364641374297200 29171309943776 2083664995984 131254487936 7177979809 333859526 12840751 392084 8911 134 1);
                my @pascal_row135 =
                  qw(1 135 9045 400995 13232835 346700277 7511839335 138432467745 2214919483920 31254974939760 393812684240976 4475144139102000 46243156104054000 437531400061434000 3812773629106782000 30756373941461374800 230672804560960311000 1614709631926722177000 10585318698186289827000 65183278299357679461000 378063014136274540873800 2070345077412932009547000 10728151764776102231289000 52707876061726067484159000 245970088288054981592742000 1092107191998964118271774480 4620453504611002038842122800 18652941926022193416066347600 71947061714657031747684483600 265459848395458703344904818800 937958130997287418485330359760 3176954959829521901321280250800 10325103619445946179294160815100 32226838569785832014160562544100 96680515709357496042481687632300 278992345332717345722590012881780 774978737035325960340527813560500 2073591755851277569559790636283500 5347684001932242153075249535678500 13300649953523781765341005255405500 31921559888457076236818412612973200 73964589985449322987749980444694000 165539796634100865734488051471458000 358027932255148384030404390391758000 748603858351673893881754634455494000 1513843358000051652071992705232221200 2961867439565318449706072684149998000 5608642598325815787741286572113826000 10282511430263995610859025382208681000 18256703968019747309076228739839903000 31401530824993965371611113432524633160 52335884708323275619351855720874388600 84542582990368368308183766933720166200 132396875249067444709042502933939128200 201047106859695008632249726677463120600 296087557375187194531131415652263868520 422982224821695992187330593788948383600 586238522121297954084195033496963549200 788389736645883455492538148495916497200 1028915419012424170727549787020094411600 1303292864082403949588229730225452921360 1602409259117709774083889012572278182000 1912552986688879407777544950489493314000 2216132825845844393139060021995762094000 2493149429076574942281442524745232355750 2723286299452874167722806450106330727050 2888333953965169571827218962233987134750 2974552877964129857553404602897688243250 2974552877964129857553404602897688243250 2888333953965169571827218962233987134750 2723286299452874167722806450106330727050 2493149429076574942281442524745232355750 2216132825845844393139060021995762094000 1912552986688879407777544950489493314000 1602409259117709774083889012572278182000 1303292864082403949588229730225452921360 1028915419012424170727549787020094411600 788389736645883455492538148495916497200 586238522121297954084195033496963549200 422982224821695992187330593788948383600 296087557375187194531131415652263868520 201047106859695008632249726677463120600 132396875249067444709042502933939128200 84542582990368368308183766933720166200 52335884708323275619351855720874388600 31401530824993965371611113432524633160 18256703968019747309076228739839903000 10282511430263995610859025382208681000 5608642598325815787741286572113826000 2961867439565318449706072684149998000 1513843358000051652071992705232221200 748603858351673893881754634455494000 358027932255148384030404390391758000 165539796634100865734488051471458000 73964589985449322987749980444694000 31921559888457076236818412612973200 13300649953523781765341005255405500 5347684001932242153075249535678500 2073591755851277569559790636283500 774978737035325960340527813560500 278992345332717345722590012881780 96680515709357496042481687632300 32226838569785832014160562544100 10325103619445946179294160815100 3176954959829521901321280250800 937958130997287418485330359760 265459848395458703344904818800 71947061714657031747684483600 18652941926022193416066347600 4620453504611002038842122800 1092107191998964118271774480 245970088288054981592742000 52707876061726067484159000 10728151764776102231289000 2070345077412932009547000 378063014136274540873800 65183278299357679461000 10585318698186289827000 1614709631926722177000 230672804560960311000 30756373941461374800 3812773629106782000 437531400061434000 46243156104054000 4475144139102000 393812684240976 31254974939760 2214919483920 138432467745 7511839335 346700277 13232835 400995 9045 135 1);
                my @pascal_row136 =
                  qw(1 136 9180 410040 13633830 359933112 7858539612 145944307080 2353351951665 33469894423680 425067659180736 4868956823342976 50718300243156000 483774556165488000 4250305029168216000 34569147570568156800 261429178502421685800 1845382436487682488000 12200028330113012004000 75768596997543969288000 443246292435632220334800 2448408091549206550420800 12798496842189034240836000 63436027826502169715448000 298677964349781049076901000 1338077280287019099864516480 5712560696609966157113897280 23273395430633195454908470400 90600003640679225163750831200 337406910110115735092589302400 1203417979392746121830235178560 4114913090826809319806610610560 13502058579275468080615441065900 42551942189231778193454723359200 128907354279143328056642250176400 375672861042074841765071700514080 1053971082368043306063117826442280 2848570492886603529900318449844000 7421275757783519722635040171962000 18648333955456023918416254791084000 45222209841980858002159417868378700 105886149873906399224568393057667200 239504386619550188722238031916152000 523567728889249249764892441863216000 1106631790606822277912159024847252000 2262447216351725545953747339687715200 4475710797565370101778065389382219200 8570510037891134237447359256263824000 15891154028589811398600311954322507000 28539215398283742919935254122048584000 49658234793013712680687342172364536160 83737415533317240990962969153399021760 136878467698691643927535622654594554800 216939458239435813017226269867659294400 333443982108762453341292229611402248800 497134664234882203163381142329726989120 719069782196883186718462009441212252120 1009220746942993946271525627285911932800 1374628258767181409576733181992880046400 1817305155658307626220087935516010908800 2332208283094828120315779517245547332960 2905702123200113723672118742797731103360 3514962245806589181861433963061771496000 4128685812534723800916604972485255408000 4709282254922419335420502546740994449750 5216435728529449110004248974851563082800 5611620253418043739550025412340317861800 5862886831929299429380623565131675378000 5949105755928259715106809205795376486500 5862886831929299429380623565131675378000 5611620253418043739550025412340317861800 5216435728529449110004248974851563082800 4709282254922419335420502546740994449750 4128685812534723800916604972485255408000 3514962245806589181861433963061771496000 2905702123200113723672118742797731103360 2332208283094828120315779517245547332960 1817305155658307626220087935516010908800 1374628258767181409576733181992880046400 1009220746942993946271525627285911932800 719069782196883186718462009441212252120 497134664234882203163381142329726989120 333443982108762453341292229611402248800 216939458239435813017226269867659294400 136878467698691643927535622654594554800 83737415533317240990962969153399021760 49658234793013712680687342172364536160 28539215398283742919935254122048584000 15891154028589811398600311954322507000 8570510037891134237447359256263824000 4475710797565370101778065389382219200 2262447216351725545953747339687715200 1106631790606822277912159024847252000 523567728889249249764892441863216000 239504386619550188722238031916152000 105886149873906399224568393057667200 45222209841980858002159417868378700 18648333955456023918416254791084000 7421275757783519722635040171962000 2848570492886603529900318449844000 1053971082368043306063117826442280 375672861042074841765071700514080 128907354279143328056642250176400 42551942189231778193454723359200 13502058579275468080615441065900 4114913090826809319806610610560 1203417979392746121830235178560 337406910110115735092589302400 90600003640679225163750831200 23273395430633195454908470400 5712560696609966157113897280 1338077280287019099864516480 298677964349781049076901000 63436027826502169715448000 12798496842189034240836000 2448408091549206550420800 443246292435632220334800 75768596997543969288000 12200028330113012004000 1845382436487682488000 261429178502421685800 34569147570568156800 4250305029168216000 483774556165488000 50718300243156000 4868956823342976 425067659180736 33469894423680 2353351951665 145944307080 7858539612 359933112 13633830 410040 9180 136 1);
                my @pascal_row137 =
                  qw(1 137 9316 419220 14043870 373566942 8218472724 153802846692 2499296258745 35823246375345 458537553604416 5294024482523712 55587257066498976 534492856408644000 4734079585333704000 38819452599736372800 295998326072989842600 2106811614990104173800 14045410766600694492000 87968625327656981292000 519014889433176189622800 2891654383984838770755600 15246904933738240791256800 76234524668691203956284000 362113992176283218792349000 1636755244636800148941417480 7050637976896985256978413760 28985956127243161612022367680 113873399071312420618659301600 428006913750794960256340133600 1540824889502861856922824480960 5318331070219555441636845789120 17616971670102277400422051676460 56054000768507246274070164425100 171459296468375106250096973535600 504580215321218169821713950690480 1429643943410118147828189526956360 3902541575254646835963436276286280 10269846250670123252535358621806000 26069609713239543641051294963046000 63870543797436881920575672659462700 151108359715887257226727810926045900 345390536493456587946806424973819200 763072115508799438487130473779368000 1630199519496071527677051466710468000 3369079006958547823865906364534967200 6738158013917095647731812729069934400 13046220835456504339225424645646043200 24461664066480945636047671210586331000 44430369426873554318535566076371091000 78197450191297455600622596294413120160 133395650326330953671650311325763557920 220615883232008884918498591807993576560 353817925938127456944761892522253849200 550383440348198266358518499479061543200 830578646343644656504673371941129237920 1216204446431765389881843151770939241240 1728290529139877132989987636727124184920 2383849005710175355848258809278791979200 3191933414425489035796821117508890955200 4149513438753135746535867452761558241760 5237910406294941843987898260043278436320 6420664369006702905533552705859502599360 7643648058341312982778038935547026904000 8837968067457143136337107519226249857750 9925717983451868445424751521592557532550 10828055981947492849554274387191880944600 11474507085347343168930648977471993239800 11811992587857559144487432770927051864500 11811992587857559144487432770927051864500 11474507085347343168930648977471993239800 10828055981947492849554274387191880944600 9925717983451868445424751521592557532550 8837968067457143136337107519226249857750 7643648058341312982778038935547026904000 6420664369006702905533552705859502599360 5237910406294941843987898260043278436320 4149513438753135746535867452761558241760 3191933414425489035796821117508890955200 2383849005710175355848258809278791979200 1728290529139877132989987636727124184920 1216204446431765389881843151770939241240 830578646343644656504673371941129237920 550383440348198266358518499479061543200 353817925938127456944761892522253849200 220615883232008884918498591807993576560 133395650326330953671650311325763557920 78197450191297455600622596294413120160 44430369426873554318535566076371091000 24461664066480945636047671210586331000 13046220835456504339225424645646043200 6738158013917095647731812729069934400 3369079006958547823865906364534967200 1630199519496071527677051466710468000 763072115508799438487130473779368000 345390536493456587946806424973819200 151108359715887257226727810926045900 63870543797436881920575672659462700 26069609713239543641051294963046000 10269846250670123252535358621806000 3902541575254646835963436276286280 1429643943410118147828189526956360 504580215321218169821713950690480 171459296468375106250096973535600 56054000768507246274070164425100 17616971670102277400422051676460 5318331070219555441636845789120 1540824889502861856922824480960 428006913750794960256340133600 113873399071312420618659301600 28985956127243161612022367680 7050637976896985256978413760 1636755244636800148941417480 362113992176283218792349000 76234524668691203956284000 15246904933738240791256800 2891654383984838770755600 519014889433176189622800 87968625327656981292000 14045410766600694492000 2106811614990104173800 295998326072989842600 38819452599736372800 4734079585333704000 534492856408644000 55587257066498976 5294024482523712 458537553604416 35823246375345 2499296258745 153802846692 8218472724 373566942 14043870 419220 9316 137 1);
                my @pascal_row138 =
                  qw(1 138 9453 428536 14463090 387610812 8592039666 162021319416 2653099105437 38322542634090 494360799979761 5752562036128128 60881281549022688 590080113475142976 5268572441742348000 43553532185070076800 334817778672726215400 2402809941063094016400 16152222381590798665800 102014036094257675784000 606983514760833170914800 3410669273418014960378400 18138559317723079562012400 91481429602429444747540800 438348516844974422748633000 1998869236813083367733766480 8687393221533785405919831240 36036594104140146869000781440 142859355198555582230681669280 541880312822107380874999435200 1968831803253656817179164614560 6859155959722417298559670270080 22935302740321832842058897465580 73670972438609523674492216101560 227513297236882352524167137960700 676039511789593276071810924226080 1934224158731336317649903477646840 5332185518664764983791625803242640 14172387825924770088498794898092280 36339455963909666893586653584852000 89940153510676425561626967622508700 214978903513324139147303483585508600 496498896209343845173534235899865100 1108462652002256026433936898753187200 2393271635004870966164181940489836000 4999278526454619351542957831245435200 10107237020875643471597719093604901600 19784378849373599986957237374715977600 37507884901937449975273095856232374200 68892033493354499954583237286957422000 122627819618171009919158162370784211160 211593100517628409272272907620176678080 354011533558339838590148903133757134480 574433809170136341863260484330247425760 904201366286325723303280392001315392400 1380962086691842922863191871420190781120 2046783092775410046386516523712068479160 2944494975571642522871830788498063426160 4112139534850052488838246446005916164120 5575782420135664391645079926787682934400 7341446853178624782332688570270449196960 9387423845048077590523765712804836678080 11658574775301644749521450965902781035680 14064312427348015888311591641406529503360 16481616125798456119115146454773276761750 18763686050909011581761859040818807390300 20753773965399361294979025908784438477150 22302563067294836018484923364663874184400 23286499673204902313418081748399045104300 23623985175715118288974865541854103729000 23286499673204902313418081748399045104300 22302563067294836018484923364663874184400 20753773965399361294979025908784438477150 18763686050909011581761859040818807390300 16481616125798456119115146454773276761750 14064312427348015888311591641406529503360 11658574775301644749521450965902781035680 9387423845048077590523765712804836678080 7341446853178624782332688570270449196960 5575782420135664391645079926787682934400 4112139534850052488838246446005916164120 2944494975571642522871830788498063426160 2046783092775410046386516523712068479160 1380962086691842922863191871420190781120 904201366286325723303280392001315392400 574433809170136341863260484330247425760 354011533558339838590148903133757134480 211593100517628409272272907620176678080 122627819618171009919158162370784211160 68892033493354499954583237286957422000 37507884901937449975273095856232374200 19784378849373599986957237374715977600 10107237020875643471597719093604901600 4999278526454619351542957831245435200 2393271635004870966164181940489836000 1108462652002256026433936898753187200 496498896209343845173534235899865100 214978903513324139147303483585508600 89940153510676425561626967622508700 36339455963909666893586653584852000 14172387825924770088498794898092280 5332185518664764983791625803242640 1934224158731336317649903477646840 676039511789593276071810924226080 227513297236882352524167137960700 73670972438609523674492216101560 22935302740321832842058897465580 6859155959722417298559670270080 1968831803253656817179164614560 541880312822107380874999435200 142859355198555582230681669280 36036594104140146869000781440 8687393221533785405919831240 1998869236813083367733766480 438348516844974422748633000 91481429602429444747540800 18138559317723079562012400 3410669273418014960378400 606983514760833170914800 102014036094257675784000 16152222381590798665800 2402809941063094016400 334817778672726215400 43553532185070076800 5268572441742348000 590080113475142976 60881281549022688 5752562036128128 494360799979761 38322542634090 2653099105437 162021319416 8592039666 387610812 14463090 428536 9453 138 1);
                my @pascal_row139 =
                  qw(1 139 9591 437989 14891626 402073902 8979650478 170613359082 2815120424853 40975641739527 532683342613851 6246922836107889 66633843585150816 650961395024165664 5858652555217490976 48822104626812424800 378371310857796292200 2737627719735820231800 18555032322653892682200 118166258475848474449800 708997550855090846698800 4017652788178848131293200 21549228591141094522390800 109619988920152524309553200 529829946447403867496173800 2437217753658057790482399480 10686262458346868773653597720 44723987325673932274920612680 178895949302695729099682450720 684739668020662963105681104480 2510712116075764198054164049760 8827987762976074115738834884640 29794458700044250140618567735660 96606275178931356516551113567140 301184269675491876198659354062260 903552809026475628595978062186780 2610263670520929593721714401872920 7266409677396101301441529280889480 19504573344589535072290420701334920 50511843789834436982085448482944280 126279609474586092455213621207360700 304919057024000564708930451208017300 711477799722667984320837719485373700 1604961548211599871607471134653052300 3501734287007126992598118839243023200 7392550161459490317707139771735271200 15106515547330262823140676924850336800 29891615870249243458554956468320879200 57292263751311049962230333230948351800 106399918395291949929856333143189796200 191519853111525509873741399657741633160 334220920135799419191431069990960889240 565604634075968247862421810753933812560 928445342728476180453409387464004560240 1478635175456462065166540876331562818160 2285163452978168646166472263421506173520 3427745179467252969249708395132259260280 4991278068347052569258347312210131905320 7056634510421695011710077234503979590280 9687921954985716880483326372793599098520 12917229273314289173977768497058132131360 16728870698226702372856454283075285875040 21045998620349722340045216678707617713760 25722887202649660637833042607309310539040 30545928553146472007426738096179806265110 35245302176707467700877005495592084152050 39517460016308372876740884949603245867450 43056337032694197313463949273448312661550 45589062740499738331903005113062919288700 46910484848920020602392947290253148833300 46910484848920020602392947290253148833300 45589062740499738331903005113062919288700 43056337032694197313463949273448312661550 39517460016308372876740884949603245867450 35245302176707467700877005495592084152050 30545928553146472007426738096179806265110 25722887202649660637833042607309310539040 21045998620349722340045216678707617713760 16728870698226702372856454283075285875040 12917229273314289173977768497058132131360 9687921954985716880483326372793599098520 7056634510421695011710077234503979590280 4991278068347052569258347312210131905320 3427745179467252969249708395132259260280 2285163452978168646166472263421506173520 1478635175456462065166540876331562818160 928445342728476180453409387464004560240 565604634075968247862421810753933812560 334220920135799419191431069990960889240 191519853111525509873741399657741633160 106399918395291949929856333143189796200 57292263751311049962230333230948351800 29891615870249243458554956468320879200 15106515547330262823140676924850336800 7392550161459490317707139771735271200 3501734287007126992598118839243023200 1604961548211599871607471134653052300 711477799722667984320837719485373700 304919057024000564708930451208017300 126279609474586092455213621207360700 50511843789834436982085448482944280 19504573344589535072290420701334920 7266409677396101301441529280889480 2610263670520929593721714401872920 903552809026475628595978062186780 301184269675491876198659354062260 96606275178931356516551113567140 29794458700044250140618567735660 8827987762976074115738834884640 2510712116075764198054164049760 684739668020662963105681104480 178895949302695729099682450720 44723987325673932274920612680 10686262458346868773653597720 2437217753658057790482399480 529829946447403867496173800 109619988920152524309553200 21549228591141094522390800 4017652788178848131293200 708997550855090846698800 118166258475848474449800 18555032322653892682200 2737627719735820231800 378371310857796292200 48822104626812424800 5858652555217490976 650961395024165664 66633843585150816 6246922836107889 532683342613851 40975641739527 2815120424853 170613359082 8979650478 402073902 14891626 437989 9591 139 1);
                my @pascal_row140 =
                  qw(1 140 9730 447580 15329615 416965528 9381724380 179593009560 2985733783935 43790762164380 573658984353378 6779606178721740 72880766421258705 717595238609316480 6509613950241656640 54680757182029915776 427193415484608717000 3115999030593616524000 21292660042389712914000 136721290798502367132000 827163809330939321148600 4726650339033938977992000 25566881379319942653684000 131169217511293618831944000 639449935367556391805727000 2967047700105461657978573280 13123480212004926564135997200 55410249784020801048574210400 223619936628369661374603063400 863635617323358692205363555200 3195451784096427161159845154240 11338699879051838313792998934400 38622446463020324256357402620300 126400733878975606657169681302800 397790544854423232715210467629400 1204737078701967504794637416249040 3513816479547405222317692464059700 9876673347917030895163243682762400 26770983021985636373731949982224400 70016417134423972054375869184279200 176791453264420529437299069690304980 431198666498586657164144072415378000 1016396856746668549029768170693391000 2316439347934267855928308854138426000 5106695835218726864205589973896075500 10894284448466617310305258610978294400 22499065708789753140847816696585608000 44998131417579506281695633393171216000 87183879621560293420785289699269231000 163692182146602999892086666374138148000 297919771506817459803597732800931429360 525740773247324929065172469648702522400 899825554211767667053852880744894701800 1494049976804444428315831198217938372800 2407080518184938245619950263795567378400 3763798628434630711333013139753068991680 5712908632445421615416180658553765433800 8419023247814305538508055707342391165600 12047912578768747580968424546714111495600 16744556465407411892193403607297578688800 22605151228300006054461094869851731229880 29646099971540991546834222780133418006400 37774869318576424712901670961782903588800 46768885822999382977878259286016928252800 56268815755796132645259780703489116804150 65791230729853939708303743591771890417160 74762762193015840577617890445195330019500 82573797049002570190204834223051558529000 88645399773193935645366954386511231950250 92499547589419758934295952403316068122000 93820969697840041204785894580506297666600 92499547589419758934295952403316068122000 88645399773193935645366954386511231950250 82573797049002570190204834223051558529000 74762762193015840577617890445195330019500 65791230729853939708303743591771890417160 56268815755796132645259780703489116804150 46768885822999382977878259286016928252800 37774869318576424712901670961782903588800 29646099971540991546834222780133418006400 22605151228300006054461094869851731229880 16744556465407411892193403607297578688800 12047912578768747580968424546714111495600 8419023247814305538508055707342391165600 5712908632445421615416180658553765433800 3763798628434630711333013139753068991680 2407080518184938245619950263795567378400 1494049976804444428315831198217938372800 899825554211767667053852880744894701800 525740773247324929065172469648702522400 297919771506817459803597732800931429360 163692182146602999892086666374138148000 87183879621560293420785289699269231000 44998131417579506281695633393171216000 22499065708789753140847816696585608000 10894284448466617310305258610978294400 5106695835218726864205589973896075500 2316439347934267855928308854138426000 1016396856746668549029768170693391000 431198666498586657164144072415378000 176791453264420529437299069690304980 70016417134423972054375869184279200 26770983021985636373731949982224400 9876673347917030895163243682762400 3513816479547405222317692464059700 1204737078701967504794637416249040 397790544854423232715210467629400 126400733878975606657169681302800 38622446463020324256357402620300 11338699879051838313792998934400 3195451784096427161159845154240 863635617323358692205363555200 223619936628369661374603063400 55410249784020801048574210400 13123480212004926564135997200 2967047700105461657978573280 639449935367556391805727000 131169217511293618831944000 25566881379319942653684000 4726650339033938977992000 827163809330939321148600 136721290798502367132000 21292660042389712914000 3115999030593616524000 427193415484608717000 54680757182029915776 6509613950241656640 717595238609316480 72880766421258705 6779606178721740 573658984353378 43790762164380 2985733783935 179593009560 9381724380 416965528 15329615 447580 9730 140 1);
                my @pascal_row141 =
                  qw(1 141 9870 457310 15777195 432295143 9798689908 188974733940 3165326793495 46776495948315 617449746517758 7353265163075118 79660372599980445 790476005030575185 7227209188850973120 61190371132271572416 481874172666638632776 3543192446078225241000 24408659072983329438000 158013950840892080046000 963885100129441688280600 5553814148364878299140600 30293531718353881631676000 156736098890613561485628000 770619152878850010637671000 3606497635473018049784300280 16090527912110388222114570480 68533729996025727612710207600 279030186412390462423177273800 1087255553951728353579966618600 4059087401419785853365208709440 14534151663148265474952844088640 49961146342072162570150401554700 165023180341995930913527083923100 524191278733398839372380148932200 1602527623556390737509847883878440 4718553558249372727112329880308740 13390489827464436117480936146822100 36647656369902667268895193664986800 96787400156409608428107819166503600 246807870398844501491674938874584180 607990119763007186601443142105682980 1447595523245255206193912243108769000 3332836204680936404958077024831817000 7423135183152994720133898828034501500 16000980283685344174510848584874369900 33393350157256370451153075307563902400 67497197126369259422543450089756824000 132182011039139799702480923092440447000 250876061768163293312871956073407379000 461611953653420459695684399175069577360 823660544754142388868770202449633951760 1425566327459092596119025350393597224200 2393875531016212095369684078962833074600 3901130494989382673935781462013505751200 6170879146619568956952963403548636370080 9476707260880052326749193798306834425480 14131931880259727153924236365896156599400 20466935826583053119476480254056502661200 28792469044176159473161828154011690184400 39349707693707417946654498477149309918680 52251251199840997601295317649985149236280 67420969290117416259735893741916321595200 84543755141575807690779930247799831841600 103037701578795515623138039989506045057000 122060046485650072353563524295261007221300 140553992922869780285921634036967220436700 157336559242018410767822724668246888548500 171219196822196505835571788609562790479200 181144947362613694579662906789827300072200 186320517287259800139081846983822365788600 186320517287259800139081846983822365788600 181144947362613694579662906789827300072200 171219196822196505835571788609562790479200 157336559242018410767822724668246888548500 140553992922869780285921634036967220436700 122060046485650072353563524295261007221300 103037701578795515623138039989506045057000 84543755141575807690779930247799831841600 67420969290117416259735893741916321595200 52251251199840997601295317649985149236280 39349707693707417946654498477149309918680 28792469044176159473161828154011690184400 20466935826583053119476480254056502661200 14131931880259727153924236365896156599400 9476707260880052326749193798306834425480 6170879146619568956952963403548636370080 3901130494989382673935781462013505751200 2393875531016212095369684078962833074600 1425566327459092596119025350393597224200 823660544754142388868770202449633951760 461611953653420459695684399175069577360 250876061768163293312871956073407379000 132182011039139799702480923092440447000 67497197126369259422543450089756824000 33393350157256370451153075307563902400 16000980283685344174510848584874369900 7423135183152994720133898828034501500 3332836204680936404958077024831817000 1447595523245255206193912243108769000 607990119763007186601443142105682980 246807870398844501491674938874584180 96787400156409608428107819166503600 36647656369902667268895193664986800 13390489827464436117480936146822100 4718553558249372727112329880308740 1602527623556390737509847883878440 524191278733398839372380148932200 165023180341995930913527083923100 49961146342072162570150401554700 14534151663148265474952844088640 4059087401419785853365208709440 1087255553951728353579966618600 279030186412390462423177273800 68533729996025727612710207600 16090527912110388222114570480 3606497635473018049784300280 770619152878850010637671000 156736098890613561485628000 30293531718353881631676000 5553814148364878299140600 963885100129441688280600 158013950840892080046000 24408659072983329438000 3543192446078225241000 481874172666638632776 61190371132271572416 7227209188850973120 790476005030575185 79660372599980445 7353265163075118 617449746517758 46776495948315 3165326793495 188974733940 9798689908 432295143 15777195 457310 9870 141 1);
                my @pascal_row142 =
                  qw(1 142 10011 467180 16234505 448072338 10230985051 198773423848 3354301527435 49941822741810 664226242466073 7970714909592876 87013637763055563 870136377630555630 8017685193881548305 68417580321122545536 543064543798910205192 4025066618744863873776 27951851519061554679000 182422609913875409484000 1121899050970333768326600 6517699248494319987421200 35847345866718759930816600 187029630608967443117304000 927355251769463572123299000 4377116788351868060421971280 19697025547583406271898870760 84624257908136115834824778080 347563916408416190035887481400 1366285740364118816003143892400 5146342955371514206945175328040 18593239064568051328318052798080 64495298005220428045103245643340 214984326684068093483677485477800 689214459075394770285907232855300 2126718902289789576882228032810640 6321081181805763464622177764187180 18109043385713808844593266027130840 50038146197367103386376129811808900 133435056526312275697003012831490400 343595270555254109919782758041087780 854797990161851688093118080980267160 2055585643008262392795355385214451980 4780431727926191611151989267940586000 10755971387833931125091975852866318500 23424115466838338894644747412908871400 49394330440941714625663923892438272300 100890547283625629873696525397320726400 199679208165509059125024373182197271000 383058072807303093015352879165847826000 712488015421583753008556355248476956360 1285272498407562848564454601624703529120 2249226872213234984987795552843231175960 3819441858475304691488709429356430298800 6295006026005594769305465540976338825800 10072009641608951630888744865562142121280 15647586407499621283702157201855470795560 23608639141139779480673430164202991024880 34598867706842780273400716619952659260600 49259404870759212592638308408068192845600 68142176737883577419816326631161000103080 91600958893548415547949816127134459154960 119672220489958413861031211391901470831500 151964724431693223950515823989716153436800 187581456720371323313917970237305876898600 225097748064445587976701564284767052278300 262614039408519852639485158332228227658000 297890552164888191053744358705214108985200 328555756064214916603394513277809679027800 352364144184810200415234695399390090551500 367465464649873494718744753773649665860800 372641034574519600278163693967644731577200 367465464649873494718744753773649665860800 352364144184810200415234695399390090551500 328555756064214916603394513277809679027800 297890552164888191053744358705214108985200 262614039408519852639485158332228227658000 225097748064445587976701564284767052278300 187581456720371323313917970237305876898600 151964724431693223950515823989716153436800 119672220489958413861031211391901470831500 91600958893548415547949816127134459154960 68142176737883577419816326631161000103080 49259404870759212592638308408068192845600 34598867706842780273400716619952659260600 23608639141139779480673430164202991024880 15647586407499621283702157201855470795560 10072009641608951630888744865562142121280 6295006026005594769305465540976338825800 3819441858475304691488709429356430298800 2249226872213234984987795552843231175960 1285272498407562848564454601624703529120 712488015421583753008556355248476956360 383058072807303093015352879165847826000 199679208165509059125024373182197271000 100890547283625629873696525397320726400 49394330440941714625663923892438272300 23424115466838338894644747412908871400 10755971387833931125091975852866318500 4780431727926191611151989267940586000 2055585643008262392795355385214451980 854797990161851688093118080980267160 343595270555254109919782758041087780 133435056526312275697003012831490400 50038146197367103386376129811808900 18109043385713808844593266027130840 6321081181805763464622177764187180 2126718902289789576882228032810640 689214459075394770285907232855300 214984326684068093483677485477800 64495298005220428045103245643340 18593239064568051328318052798080 5146342955371514206945175328040 1366285740364118816003143892400 347563916408416190035887481400 84624257908136115834824778080 19697025547583406271898870760 4377116788351868060421971280 927355251769463572123299000 187029630608967443117304000 35847345866718759930816600 6517699248494319987421200 1121899050970333768326600 182422609913875409484000 27951851519061554679000 4025066618744863873776 543064543798910205192 68417580321122545536 8017685193881548305 870136377630555630 87013637763055563 7970714909592876 664226242466073 49941822741810 3354301527435 198773423848 10230985051 448072338 16234505 467180 10011 142 1);
                my @pascal_row143 =
                  qw(1 143 10153 477191 16701685 464306843 10679057389 209004408899 3553074951283 53296124269245 714168065207883 8634941152058949 94984352672648439 957150015393611193 8887821571512103935 76435265515004093841 611482124120032750728 4568131162543774078968 31976918137806418552776 210374461432936964163000 1304321660884209177810600 7639598299464653755747800 42365045115213079918237800 222876976475686203048120600 1114384882378431015240603000 5304472040121331632545270280 24074142335935274332320842040 104321283455719522106723648840 432188174316552305870712259480 1713849656772535006039031373800 6512628695735633022948319220440 23739582019939565535263228126120 83088537069788479373421298441420 279479624689288521528780731121140 904198785759462863769584718333100 2815933361365184347168135265665940 8447800084095553041504405796997820 24430124567519572309215443791318020 68147189583080912230969395838939740 183473202723679379083379142643299300 477030327081566385616785770872578180 1198393260717105798012900839021354940 2910383633170114080888473466194719140 6836017370934454003947344653155037980 15536403115760122736243965120806904500 34180086854672270019736723265775189900 72818445907780053520308671305347143700 150284877724567344499360449289758998700 300569755449134688998720898579517997400 582737280972812152140377252348045097000 1095546088228886846023909234414324782360 1997760513829146601573010956873180485480 3534499370620797833552250154467934705080 6068668730688539676476504982199661474760 10114447884480899460794174970332769124600 16367015667614546400194210406538480947080 25719596049108572914590902067417612916840 39256225548639400764375587366058461820440 58207506847982559754074146784155650285480 83858272577601992866039025028020852106200 117401581608642790012454635039229192948700 159743135631431992967766142758295459258000 211273179383506829408981027519035929986400 271636944921651637811547035381617624268300 339546181152064547264433794227022030335400 412679204784816911290619534522072929176800 487711787472965440616186722616995279936200 560504591573408043693229517037442336643100 626446308229103107657138871983023788012900 680919900249025117018629208677199769579200 719829608834683695133979449173039756412400 740106499224393094996908447741294397438000 740106499224393094996908447741294397438000 719829608834683695133979449173039756412400 680919900249025117018629208677199769579200 626446308229103107657138871983023788012900 560504591573408043693229517037442336643100 487711787472965440616186722616995279936200 412679204784816911290619534522072929176800 339546181152064547264433794227022030335400 271636944921651637811547035381617624268300 211273179383506829408981027519035929986400 159743135631431992967766142758295459258000 117401581608642790012454635039229192948700 83858272577601992866039025028020852106200 58207506847982559754074146784155650285480 39256225548639400764375587366058461820440 25719596049108572914590902067417612916840 16367015667614546400194210406538480947080 10114447884480899460794174970332769124600 6068668730688539676476504982199661474760 3534499370620797833552250154467934705080 1997760513829146601573010956873180485480 1095546088228886846023909234414324782360 582737280972812152140377252348045097000 300569755449134688998720898579517997400 150284877724567344499360449289758998700 72818445907780053520308671305347143700 34180086854672270019736723265775189900 15536403115760122736243965120806904500 6836017370934454003947344653155037980 2910383633170114080888473466194719140 1198393260717105798012900839021354940 477030327081566385616785770872578180 183473202723679379083379142643299300 68147189583080912230969395838939740 24430124567519572309215443791318020 8447800084095553041504405796997820 2815933361365184347168135265665940 904198785759462863769584718333100 279479624689288521528780731121140 83088537069788479373421298441420 23739582019939565535263228126120 6512628695735633022948319220440 1713849656772535006039031373800 432188174316552305870712259480 104321283455719522106723648840 24074142335935274332320842040 5304472040121331632545270280 1114384882378431015240603000 222876976475686203048120600 42365045115213079918237800 7639598299464653755747800 1304321660884209177810600 210374461432936964163000 31976918137806418552776 4568131162543774078968 611482124120032750728 76435265515004093841 8887821571512103935 957150015393611193 94984352672648439 8634941152058949 714168065207883 53296124269245 3553074951283 209004408899 10679057389 464306843 16701685 477191 10153 143 1);
                my @pascal_row144 =
                  qw(1 144 10296 487344 17178876 481008528 11143364232 219683466288 3762079360182 56849199220528 767464189477128 9349109217266832 103619293824707388 1052134368066259632 9844971586905715128 85323087086516197776 687917389635036844569 5179613286663806829696 36545049300350192631744 242351379570743382715776 1514696122317146141973600 8943919960348862933558400 50004643414677733673985600 265242021590899282966358400 1337261858854117218288723600 6418856922499762647785873280 29378614376056605964866112320 128395425791654796439044490880 536509457772271827977435908320 2146037831089087311909743633280 8226478352508168028987350594240 30252210715675198558211547346560 106828119089728044908684526567540 362568161759077000902202029562560 1183678410448751385298365449454240 3720132147124647210937719983999040 11263733445460737388672541062663760 32877924651615125350719849588315840 92577314150600484540184839630257760 251620392306760291314348538482239040 660503529805245764700164913515877480 1675423587798672183629686609893933120 4108776893887219878901374305216074080 9746401004104568084835818119349757120 22372420486694576740191309773961942480 49716489970432392755980688386582094400 106998532762452323540045394571122333600 223103323632347398019669120595106142400 450854633173702033498081347869276996100 883307036421946841139098150927563094400 1678283369201698998164286486762369879360 3093306602058033447596920191287505267840 5532259884449944435125261111341115190560 9603168101309337510028755136667596179840 16183116615169439137270679952532430599360 26481463552095445860988385376871250071680 42086611716723119314785112473956093863920 64975821597747973678966489433476074737280 97463732396621960518449734150214112105920 142065779425584552620113171812176502391700 201259854186244782878493660067250045054900 277144717240074782980220777797524652206700 371016315014938822376747170277331389244500 482910124305158467220528062900653554254700 611183126073716185075980829608639654603600 752225385936881458555053328749094959512200 900390992257782351906806257139068209113000 1048216379046373484309416239654437616579000 1186950899802511151350368389020466124656000 1307366208478128224675768080660223557592000 1400749509083708812152608657850239525992000 1459936108059076790130887896914334153850000 1480212998448786189993816895482588794876000 1459936108059076790130887896914334153850000 1400749509083708812152608657850239525992000 1307366208478128224675768080660223557592000 1186950899802511151350368389020466124656000 1048216379046373484309416239654437616579000 900390992257782351906806257139068209113000 752225385936881458555053328749094959512200 611183126073716185075980829608639654603600 482910124305158467220528062900653554254700 371016315014938822376747170277331389244500 277144717240074782980220777797524652206700 201259854186244782878493660067250045054900 142065779425584552620113171812176502391700 97463732396621960518449734150214112105920 64975821597747973678966489433476074737280 42086611716723119314785112473956093863920 26481463552095445860988385376871250071680 16183116615169439137270679952532430599360 9603168101309337510028755136667596179840 5532259884449944435125261111341115190560 3093306602058033447596920191287505267840 1678283369201698998164286486762369879360 883307036421946841139098150927563094400 450854633173702033498081347869276996100 223103323632347398019669120595106142400 106998532762452323540045394571122333600 49716489970432392755980688386582094400 22372420486694576740191309773961942480 9746401004104568084835818119349757120 4108776893887219878901374305216074080 1675423587798672183629686609893933120 660503529805245764700164913515877480 251620392306760291314348538482239040 92577314150600484540184839630257760 32877924651615125350719849588315840 11263733445460737388672541062663760 3720132147124647210937719983999040 1183678410448751385298365449454240 362568161759077000902202029562560 106828119089728044908684526567540 30252210715675198558211547346560 8226478352508168028987350594240 2146037831089087311909743633280 536509457772271827977435908320 128395425791654796439044490880 29378614376056605964866112320 6418856922499762647785873280 1337261858854117218288723600 265242021590899282966358400 50004643414677733673985600 8943919960348862933558400 1514696122317146141973600 242351379570743382715776 36545049300350192631744 5179613286663806829696 687917389635036844569 85323087086516197776 9844971586905715128 1052134368066259632 103619293824707388 9349109217266832 767464189477128 56849199220528 3762079360182 219683466288 11143364232 481008528 17178876 487344 10296 144 1);
                my @pascal_row145 =
                  qw(1 145 10440 497640 17666220 498187404 11624372760 230826830520 3981762826470 60611278580710 824313388697656 10116573406743960 112968403041974220 1155753661890967020 10897105954971974760 95168058673421912904 773240476721553042345 5867530676298843674265 41724662587013999461440 278896428871093575347520 1757047501887889524689376 10458616082666009075532000 58948563375026596607544000 315246665005577016640344000 1602503880445016501255082000 7756118781353879866074596880 35797471298556368612651985600 157774040167711402403910603200 664904883563926624416480399200 2682547288861359139887179541600 10372516183597255340897094227520 38478689068183366587198897940800 137080329805403243466896073914100 469396280848805045810886556130100 1546246572207828386200567479016800 4903810557573398596236085433453280 14983865592585384599610261046662800 44141658097075862739392390650979600 125455238802215609890904689218573600 344197706457360775854533378112496800 912123922112006056014513451998116520 2335927117603917948329851523409810600 5784200481685892062531060915110007200 13855177897991787963737192424565831200 32118821490799144825027127893311699600 72088910457126969496171998160544036880 156715022732884716296026082957704428000 330101856394799721559714515166228476000 673957956806049431517750468464383138500 1334161669595648874637179498796840090500 2561590405623645839303384637689932973760 4771589971259732445761206678049875147200 8625566486507977882722181302628620458400 15135427985759281945154016248008711370400 25786284716478776647299435089200026779200 42664580167264884998259065329403680671040 68568075268818565175773497850827343935600 107062433314471092993751601907432168601200 162439553994369934197416223583690186843200 239529511822206513138562905962390614497600 343325633611829335498606831879426547446600 478404571426319565858714437864774697261600 648161032255013605356967948074856041451200 853926439320097289597275233177984943499200 1094093250378874652296508892509293208858000 1363408512010597643631034158357734614116000 1652616378194663810461859585888163168625000 1948607371304155836216222496793505825692000 2235167278848884635659784628674903741235000 2494317108280639376026136469680689682248000 2708115717561837036828376738510463083584000 2860685617142785602283496554764573679842000 2940149106507862980124704792396922948726000 2940149106507862980124704792396922948726000 2860685617142785602283496554764573679842000 2708115717561837036828376738510463083584000 2494317108280639376026136469680689682248000 2235167278848884635659784628674903741235000 1948607371304155836216222496793505825692000 1652616378194663810461859585888163168625000 1363408512010597643631034158357734614116000 1094093250378874652296508892509293208858000 853926439320097289597275233177984943499200 648161032255013605356967948074856041451200 478404571426319565858714437864774697261600 343325633611829335498606831879426547446600 239529511822206513138562905962390614497600 162439553994369934197416223583690186843200 107062433314471092993751601907432168601200 68568075268818565175773497850827343935600 42664580167264884998259065329403680671040 25786284716478776647299435089200026779200 15135427985759281945154016248008711370400 8625566486507977882722181302628620458400 4771589971259732445761206678049875147200 2561590405623645839303384637689932973760 1334161669595648874637179498796840090500 673957956806049431517750468464383138500 330101856394799721559714515166228476000 156715022732884716296026082957704428000 72088910457126969496171998160544036880 32118821490799144825027127893311699600 13855177897991787963737192424565831200 5784200481685892062531060915110007200 2335927117603917948329851523409810600 912123922112006056014513451998116520 344197706457360775854533378112496800 125455238802215609890904689218573600 44141658097075862739392390650979600 14983865592585384599610261046662800 4903810557573398596236085433453280 1546246572207828386200567479016800 469396280848805045810886556130100 137080329805403243466896073914100 38478689068183366587198897940800 10372516183597255340897094227520 2682547288861359139887179541600 664904883563926624416480399200 157774040167711402403910603200 35797471298556368612651985600 7756118781353879866074596880 1602503880445016501255082000 315246665005577016640344000 58948563375026596607544000 10458616082666009075532000 1757047501887889524689376 278896428871093575347520 41724662587013999461440 5867530676298843674265 773240476721553042345 95168058673421912904 10897105954971974760 1155753661890967020 112968403041974220 10116573406743960 824313388697656 60611278580710 3981762826470 230826830520 11624372760 498187404 17666220 497640 10440 145 1);
                my @pascal_row146 =
                  qw(1 146 10585 508080 18163860 515853624 12122560164 242451203280 4212589656990 64593041407180 884924667278366 10940886795441616 123084976448718180 1268722064932941240 12052859616862941780 106065164628393887664 868408535394974955249 6640771153020396716610 47592193263312843135705 320621091458107574808960 2035943930758983100036896 12215663584553898600221376 69407179457692605683076000 374195228380603613247888000 1917750545450593517895426000 9358622661798896367329678880 43553590079910248478726582480 193571511466267771016562588800 822678923731638026820391002400 3347452172425285764303659940800 13055063472458614480784273769120 48851205251780621928095992168320 175559018873586610054094971854900 606476610654208289277782630044200 2015642853056633432011454035146900 6450057129781226982436652912470080 19887676150158783195846346480116080 59125523689661247339002651697642400 169596896899291472630297079869553200 469652945259576385745438067331070400 1256321628569366831869046830110613320 3248051039715924004344364975407927120 8120127599289810010860912438519817800 19639378379677680026268253339675838400 45973999388790932788764320317877530800 104207731947926114321199126053855736480 228803933190011685792198081118248464880 486816879127684437855740598123932904000 1004059813200849153077464983630611614500 2008119626401698306154929967261223229000 3895752075219294713940564136486773064260 7333180376883378285064591315739808120960 13397156457767710328483387980678495605600 23760994472267259827876197550637331828800 40921712702238058592453451337208738149600 68450864883743661645558500418603707450240 111232655436083450174032563180231024606600 175630508583289658169525099758259512536800 269501987308841027191167825491122355444400 401969065816576447335979129546080801340800 582855145434035848637169737841817161944200 821730205038148901357321269744201244708200 1126565603681333171215682385939630738713000 1502087471575110894954243181252840984950000 1948019689698971941893784125687278152358000 2457501762389472295927543050867027822974000 3016024890205261454092893744245897782741000 3601223749498819646678082082681668994318000 4183774650153040471876007125468409566928000 4729484387129524011685921098355593423484000 5202432825842476412854513208191152765832000 5568801334704622639111873293275036763426000 5800834723650648582408201347161496628568000 5880298213015725960249409584793845897453000 5800834723650648582408201347161496628568000 5568801334704622639111873293275036763426000 5202432825842476412854513208191152765832000 4729484387129524011685921098355593423484000 4183774650153040471876007125468409566928000 3601223749498819646678082082681668994318000 3016024890205261454092893744245897782741000 2457501762389472295927543050867027822974000 1948019689698971941893784125687278152358000 1502087471575110894954243181252840984950000 1126565603681333171215682385939630738713000 821730205038148901357321269744201244708200 582855145434035848637169737841817161944200 401969065816576447335979129546080801340800 269501987308841027191167825491122355444400 175630508583289658169525099758259512536800 111232655436083450174032563180231024606600 68450864883743661645558500418603707450240 40921712702238058592453451337208738149600 23760994472267259827876197550637331828800 13397156457767710328483387980678495605600 7333180376883378285064591315739808120960 3895752075219294713940564136486773064260 2008119626401698306154929967261223229000 1004059813200849153077464983630611614500 486816879127684437855740598123932904000 228803933190011685792198081118248464880 104207731947926114321199126053855736480 45973999388790932788764320317877530800 19639378379677680026268253339675838400 8120127599289810010860912438519817800 3248051039715924004344364975407927120 1256321628569366831869046830110613320 469652945259576385745438067331070400 169596896899291472630297079869553200 59125523689661247339002651697642400 19887676150158783195846346480116080 6450057129781226982436652912470080 2015642853056633432011454035146900 606476610654208289277782630044200 175559018873586610054094971854900 48851205251780621928095992168320 13055063472458614480784273769120 3347452172425285764303659940800 822678923731638026820391002400 193571511466267771016562588800 43553590079910248478726582480 9358622661798896367329678880 1917750545450593517895426000 374195228380603613247888000 69407179457692605683076000 12215663584553898600221376 2035943930758983100036896 320621091458107574808960 47592193263312843135705 6640771153020396716610 868408535394974955249 106065164628393887664 12052859616862941780 1268722064932941240 123084976448718180 10940886795441616 884924667278366 64593041407180 4212589656990 242451203280 12122560164 515853624 18163860 508080 10585 146 1);
                my @pascal_row147 =
                  qw(1 147 10731 518665 18671940 534017484 12638413788 254573763444 4455040860270 68805631064170 949517708685546 11825811462719982 134025863244159796 1391807041381659420 13321581681795883020 118118024245256829444 974473700023368842913 7509179688415371671859 54232964416333239852315 368213284721420417944665 2356565022217090674845856 14251607515312881700258272 81622843042246504283297376 443602407838296218930964000 2291945773831197131143314000 11276373207249489885225104880 52912212741709144846056261360 237125101546178019495289171280 1016250435197905797836953591200 4170131096156923791124050943200 16402515644883900245087933709920 61906268724239236408880265937440 224410224125367231982190964023220 782035629527794899331877601899100 2622119463710841721289236665191100 8465699982837860414448106947616980 26337733279940010178282999392586160 79013199839820030534848998177758480 228722420588952719969299731567195600 639249842158867858375735147200623600 1725974573828943217614484897441683720 4504372668285290836213411805518540440 11368178639005734015205277413927744920 27759505978967490037129165778195656200 65613377768468612815032573657553369200 150181731336717047109963446371733267280 333011665137937800113397207172104201360 715620812317696123647938679242181368880 1490876692328533590933205581754544518500 3012179439602547459232394950891834843500 5903871701620993020095494103747996293260 11228932452102672999005155452226581185220 20730336834651088613547979296418303726560 37158150930034970156359585531315827434400 64682707174505318420329648887846069978400 109372577585981720238011951755812445599800 179683520319827111819591063598834732056900 286863164019373108343557662938490537143400 445132495892130685360692925249381867981200 671471053125417474527146955037203156785200 984824211250612295973148867387897963285000 1404585350472184749994491007586018406652000 1948295808719482072573003655683831983421000 2628653075256444066169925567192471723663000 3450107161274082836848027306940119137308000 4405521452088444237821327176554305975332000 5473526652594733750020436795112925605715000 6617248639704081100770975826927566777059000 7784998399651860118554089208150078561245000 8913259037282564483561928223824002990411000 9931917212972000424540434306546746189316000 10771234160547099051966386501466189529260000 11369636058355271221520074640436533391990000 11681132936666374542657610931955342526020000 11681132936666374542657610931955342526020000 11369636058355271221520074640436533391990000 10771234160547099051966386501466189529260000 9931917212972000424540434306546746189316000 8913259037282564483561928223824002990411000 7784998399651860118554089208150078561245000 6617248639704081100770975826927566777059000 5473526652594733750020436795112925605715000 4405521452088444237821327176554305975332000 3450107161274082836848027306940119137308000 2628653075256444066169925567192471723663000 1948295808719482072573003655683831983421000 1404585350472184749994491007586018406652000 984824211250612295973148867387897963285000 671471053125417474527146955037203156785200 445132495892130685360692925249381867981200 286863164019373108343557662938490537143400 179683520319827111819591063598834732056900 109372577585981720238011951755812445599800 64682707174505318420329648887846069978400 37158150930034970156359585531315827434400 20730336834651088613547979296418303726560 11228932452102672999005155452226581185220 5903871701620993020095494103747996293260 3012179439602547459232394950891834843500 1490876692328533590933205581754544518500 715620812317696123647938679242181368880 333011665137937800113397207172104201360 150181731336717047109963446371733267280 65613377768468612815032573657553369200 27759505978967490037129165778195656200 11368178639005734015205277413927744920 4504372668285290836213411805518540440 1725974573828943217614484897441683720 639249842158867858375735147200623600 228722420588952719969299731567195600 79013199839820030534848998177758480 26337733279940010178282999392586160 8465699982837860414448106947616980 2622119463710841721289236665191100 782035629527794899331877601899100 224410224125367231982190964023220 61906268724239236408880265937440 16402515644883900245087933709920 4170131096156923791124050943200 1016250435197905797836953591200 237125101546178019495289171280 52912212741709144846056261360 11276373207249489885225104880 2291945773831197131143314000 443602407838296218930964000 81622843042246504283297376 14251607515312881700258272 2356565022217090674845856 368213284721420417944665 54232964416333239852315 7509179688415371671859 974473700023368842913 118118024245256829444 13321581681795883020 1391807041381659420 134025863244159796 11825811462719982 949517708685546 68805631064170 4455040860270 254573763444 12638413788 534017484 18671940 518665 10731 147 1);
                my @pascal_row148 =
                  qw(1 148 10878 529396 19190605 552689424 13172431272 267212177232 4709614623714 73260671924440 1018323339749716 12775329171405528 145851674706879778 1525832904625819216 14713388723177542440 131439605927052712464 1092591724268625672357 8483653388438740514772 61742144104748611524174 422446249137753657796980 2724778306938511092790521 16608172537529972375104128 95874450557559385983555648 525225250880542723214261376 2735548181669493350074278000 13568318981080687016368418880 64188585948958634731281366240 290037314287887164341345432640 1253375536744083817332242762480 5186381531354829588961004534400 20572646741040824036211984653120 78308784369123136653968199647360 286316492849606468391071229960660 1006445853653162131314068565922320 3404155093238636620621114267090200 11087819446548702135737343612808080 34803433262777870592731106340203140 105350933119760040713131997570344640 307735620428772750504148729744954080 867972262747820578345034878767819200 2365224415987811075990220044642307320 6230347242114234053827896702960224160 15872551307291024851418689219446285360 39127684617973224052334443192123401120 93372883747436102852161739435749025400 215795109105185659924996020029286636480 483193396474654847223360653543837468640 1048632477455633923761335886414285570240 2206497504646229714581144260996725887380 4503056131931081050165600532646379362000 8916051141223540479327889054639831136760 17132804153723666019100649555974577478480 31959269286753761612553134748644884911780 57888487764686058769907564827734131160960 101840858104540288576689234419161897412800 174055284760487038658341600643658515578200 289056097905808832057603015354647177656700 466546684339200220163148726537325269200300 731995659911503793704250588187872405124600 1116603549017548159887839880286585024766000 1656295264376029770500295822425101120070000 2389409561722797045967639874973916369937000 3352881159191666822567494663269850390073000 4576948883975926138742929222876303707084000 6078760236530526903017952874132590860971000 7855628613362527074669354483494425112640000 9879048104683177987841763971667231581047000 12090775292298814850791412622040492382770000 14402247039355941219325065035077645338300000 16698257436934424602116017431974081551660000 18845176250254564908102362530370749179730000 20703151373519099476506820808012935718570000 22140870218902370273486461141902722921250000 23050768995021645764177685572391875918020000 23362265873332749085315221863910685052040000 23050768995021645764177685572391875918020000 22140870218902370273486461141902722921250000 20703151373519099476506820808012935718570000 18845176250254564908102362530370749179730000 16698257436934424602116017431974081551660000 14402247039355941219325065035077645338300000 12090775292298814850791412622040492382770000 9879048104683177987841763971667231581047000 7855628613362527074669354483494425112640000 6078760236530526903017952874132590860971000 4576948883975926138742929222876303707084000 3352881159191666822567494663269850390073000 2389409561722797045967639874973916369937000 1656295264376029770500295822425101120070000 1116603549017548159887839880286585024766000 731995659911503793704250588187872405124600 466546684339200220163148726537325269200300 289056097905808832057603015354647177656700 174055284760487038658341600643658515578200 101840858104540288576689234419161897412800 57888487764686058769907564827734131160960 31959269286753761612553134748644884911780 17132804153723666019100649555974577478480 8916051141223540479327889054639831136760 4503056131931081050165600532646379362000 2206497504646229714581144260996725887380 1048632477455633923761335886414285570240 483193396474654847223360653543837468640 215795109105185659924996020029286636480 93372883747436102852161739435749025400 39127684617973224052334443192123401120 15872551307291024851418689219446285360 6230347242114234053827896702960224160 2365224415987811075990220044642307320 867972262747820578345034878767819200 307735620428772750504148729744954080 105350933119760040713131997570344640 34803433262777870592731106340203140 11087819446548702135737343612808080 3404155093238636620621114267090200 1006445853653162131314068565922320 286316492849606468391071229960660 78308784369123136653968199647360 20572646741040824036211984653120 5186381531354829588961004534400 1253375536744083817332242762480 290037314287887164341345432640 64188585948958634731281366240 13568318981080687016368418880 2735548181669493350074278000 525225250880542723214261376 95874450557559385983555648 16608172537529972375104128 2724778306938511092790521 422446249137753657796980 61742144104748611524174 8483653388438740514772 1092591724268625672357 131439605927052712464 14713388723177542440 1525832904625819216 145851674706879778 12775329171405528 1018323339749716 73260671924440 4709614623714 267212177232 13172431272 552689424 19190605 529396 10878 148 1);
                my @pascal_row149 =
                  qw(1 149 11026 540274 19720001 571880029 13725120696 280384608504 4976826800946 77970286548154 1091584011674156 13793652511155244 158627003878285306 1671684579332698994 16239221627803361656 146152994650230254904 1224031330195678384821 9576245112707366187129 70225797493187352038946 484188393242502269321154 3147224556076264750587501 19332950844468483467894649 112482623095089358358659776 621099701438102109197817024 3260773432550036073288539376 16303867162750180366442696880 77756904930039321747649785120 354225900236845799072626798880 1543412851031970981673588195120 6439757068098913406293247296880 25759028272395653625172989187520 98881431110163960690180184300480 364625277218729605045039429608020 1292762346502768599705139795882980 4410600946891798751935182833012520 14491974539787338756358457879898280 45891252709326572728468449953011220 140154366382537911305863103910547780 413086553548532791217280727315298720 1175707883176593328849183608512773280 3233196678735631654335254923410126520 8595571658102045129818116747602531480 22102898549405258905246585922406509520 55000235925264248903753132411569686480 132500568365409326904496182627872426520 309167992852621762777157759465035661880 698988505579840507148356673573124105120 1531825873930288770984696539958123038880 3255129982101863638342480147411011457620 6709553636577310764746744793643105249380 13419107273154621529493489587286210498760 26048855294947206498428538610614408615240 49092073440477427631653784304619462390260 89847757051439820382460699576379016072740 159729345869226347346596799246896028573800 275896142865027327235030835062820412991000 463111382666295870715944615998305693235000 755602782245009052220751741891972446857000 1198542344250704013867399314725197674325000 1848599208929051953592090468474457429891000 2772898813393577930388135702711686144837000 4045704826098826816467935697399017490007000 5742290720914463868535134538243766760011000 7929830043167592961310423886146154097157000 10655709120506453041760882097008894568060000 13934388849893053977687307357627015973610000 17734676718045705062511118455161656693690000 21969823396981992838633176593707723963820000 26493022331654756070116477657118137721080000 31100504476290365821441082467051726889960000 35543433687188989510218379962344830731380000 39548327623773664384609183338383684898300000 42844021592421469749993281949915658639830000 45191639213924016037664146714294598839270000 46413034868354394849492907436302560970060000 46413034868354394849492907436302560970060000 45191639213924016037664146714294598839270000 42844021592421469749993281949915658639830000 39548327623773664384609183338383684898300000 35543433687188989510218379962344830731380000 31100504476290365821441082467051726889960000 26493022331654756070116477657118137721080000 21969823396981992838633176593707723963820000 17734676718045705062511118455161656693690000 13934388849893053977687307357627015973610000 10655709120506453041760882097008894568060000 7929830043167592961310423886146154097157000 5742290720914463868535134538243766760011000 4045704826098826816467935697399017490007000 2772898813393577930388135702711686144837000 1848599208929051953592090468474457429891000 1198542344250704013867399314725197674325000 755602782245009052220751741891972446857000 463111382666295870715944615998305693235000 275896142865027327235030835062820412991000 159729345869226347346596799246896028573800 89847757051439820382460699576379016072740 49092073440477427631653784304619462390260 26048855294947206498428538610614408615240 13419107273154621529493489587286210498760 6709553636577310764746744793643105249380 3255129982101863638342480147411011457620 1531825873930288770984696539958123038880 698988505579840507148356673573124105120 309167992852621762777157759465035661880 132500568365409326904496182627872426520 55000235925264248903753132411569686480 22102898549405258905246585922406509520 8595571658102045129818116747602531480 3233196678735631654335254923410126520 1175707883176593328849183608512773280 413086553548532791217280727315298720 140154366382537911305863103910547780 45891252709326572728468449953011220 14491974539787338756358457879898280 4410600946891798751935182833012520 1292762346502768599705139795882980 364625277218729605045039429608020 98881431110163960690180184300480 25759028272395653625172989187520 6439757068098913406293247296880 1543412851031970981673588195120 354225900236845799072626798880 77756904930039321747649785120 16303867162750180366442696880 3260773432550036073288539376 621099701438102109197817024 112482623095089358358659776 19332950844468483467894649 3147224556076264750587501 484188393242502269321154 70225797493187352038946 9576245112707366187129 1224031330195678384821 146152994650230254904 16239221627803361656 1671684579332698994 158627003878285306 13793652511155244 1091584011674156 77970286548154 4976826800946 280384608504 13725120696 571880029 19720001 540274 11026 149 1);
                my @pascal_row150 =
                  qw(1 150 11175 551300 20260275 591600030 14297000725 294109729200 5257211409450 82947113349100 1169554298222310 14885236522829400 172420656389440550 1830311583210984300 17910906207136060650 162392216278033616560 1370184324845908639725 10800276442903044571950 79802042605894718226075 554414190735689621360100 3631412949318767019908655 22480175400544748218482150 131815573939557841826554425 733582324533191467556476800 3881873133988138182486356400 19564640595300216439731236256 94060772092789502114092482000 431982805166885120820276584000 1897638751268816780746214994000 7983169919130884387966835492000 32198785340494567031466236484400 124640459382559614315353173488000 463506708328893565735219613908500 1657387623721498204750179225491000 5703363293394567351640322628895500 18902575486679137508293640712910800 60383227249113911484826907832909500 186045619091864484034331553863559000 553240919931070702523143831225846500 1588794436725126120066464335828072000 4408904561912224983184438531922899800 11828768336837676784153371671012658000 30698470207507304035064702670009041000 77103134474669507808999718333976196000 187500804290673575808249315039442113000 441668561218031089681653942092908088400 1008156498432462269925514433038159767000 2230814379510129278133053213531247144000 4786955856032152409327176687369134496500 9964683618679174403089224941054116707000 20128660909731932294240234380929315748140 39467962568101828027922028197900619114000 75140928735424634130082322915233871005500 138939830491917248014114483880998478463000 249577102920666167729057498823275044646500 435625488734253674581627634309716441564800 739007525531323197950975451061126106226000 1218714164911304922936696357890278140092000 1954145126495713066088151056617170121182000 3047141553179755967459489783199655104216000 4621498022322629883980226171186143574728000 6818603639492404746856071400110703634844000 9787995547013290685003070235642784250018000 13672120764082056829845558424389920857170000 18585539163674046003071305983155048665210000 24590097970399507019448189454635910541670000 31669065567938759040198425812788672667300000 39704500115027697901144295048869380657510000 48462845728636748908749654250825861684900000 57593526807945121891557560124169864611040000 66643938163479355331659462429396557621340000 75091761310962653894827563300728515629680000 82392349216195134134602465288299343538130000 88035660806345485787657428664210257479090000 91604674082278410887157054150597159809330000 92826069736708789698985814872605121940120000 91604674082278410887157054150597159809330000 88035660806345485787657428664210257479090000 82392349216195134134602465288299343538130000 75091761310962653894827563300728515629680000 66643938163479355331659462429396557621340000 57593526807945121891557560124169864611040000 48462845728636748908749654250825861684900000 39704500115027697901144295048869380657510000 31669065567938759040198425812788672667300000 24590097970399507019448189454635910541670000 18585539163674046003071305983155048665210000 13672120764082056829845558424389920857170000 9787995547013290685003070235642784250018000 6818603639492404746856071400110703634844000 4621498022322629883980226171186143574728000 3047141553179755967459489783199655104216000 1954145126495713066088151056617170121182000 1218714164911304922936696357890278140092000 739007525531323197950975451061126106226000 435625488734253674581627634309716441564800 249577102920666167729057498823275044646500 138939830491917248014114483880998478463000 75140928735424634130082322915233871005500 39467962568101828027922028197900619114000 20128660909731932294240234380929315748140 9964683618679174403089224941054116707000 4786955856032152409327176687369134496500 2230814379510129278133053213531247144000 1008156498432462269925514433038159767000 441668561218031089681653942092908088400 187500804290673575808249315039442113000 77103134474669507808999718333976196000 30698470207507304035064702670009041000 11828768336837676784153371671012658000 4408904561912224983184438531922899800 1588794436725126120066464335828072000 553240919931070702523143831225846500 186045619091864484034331553863559000 60383227249113911484826907832909500 18902575486679137508293640712910800 5703363293394567351640322628895500 1657387623721498204750179225491000 463506708328893565735219613908500 124640459382559614315353173488000 32198785340494567031466236484400 7983169919130884387966835492000 1897638751268816780746214994000 431982805166885120820276584000 94060772092789502114092482000 19564640595300216439731236256 3881873133988138182486356400 733582324533191467556476800 131815573939557841826554425 22480175400544748218482150 3631412949318767019908655 554414190735689621360100 79802042605894718226075 10800276442903044571950 1370184324845908639725 162392216278033616560 17910906207136060650 1830311583210984300 172420656389440550 14885236522829400 1169554298222310 82947113349100 5257211409450 294109729200 14297000725 591600030 20260275 551300 11175 150 1);
                my @nCr = (
                           \@pascal_row1,   \@pascal_row2,   \@pascal_row3,   \@pascal_row4,   \@pascal_row5,
                           \@pascal_row6,   \@pascal_row7,   \@pascal_row8,   \@pascal_row9,   \@pascal_row10,
                           \@pascal_row11,  \@pascal_row12,  \@pascal_row13,  \@pascal_row14,  \@pascal_row15,
                           \@pascal_row16,  \@pascal_row17,  \@pascal_row18,  \@pascal_row19,  \@pascal_row20,
                           \@pascal_row21,  \@pascal_row22,  \@pascal_row23,  \@pascal_row24,  \@pascal_row25,
                           \@pascal_row26,  \@pascal_row27,  \@pascal_row28,  \@pascal_row29,  \@pascal_row30,
                           \@pascal_row31,  \@pascal_row32,  \@pascal_row33,  \@pascal_row34,  \@pascal_row35,
                           \@pascal_row36,  \@pascal_row37,  \@pascal_row38,  \@pascal_row39,  \@pascal_row40,
                           \@pascal_row41,  \@pascal_row42,  \@pascal_row43,  \@pascal_row44,  \@pascal_row45,
                           \@pascal_row46,  \@pascal_row47,  \@pascal_row48,  \@pascal_row49,  \@pascal_row50,
                           \@pascal_row51,  \@pascal_row52,  \@pascal_row53,  \@pascal_row54,  \@pascal_row55,
                           \@pascal_row56,  \@pascal_row57,  \@pascal_row58,  \@pascal_row59,  \@pascal_row60,
                           \@pascal_row61,  \@pascal_row62,  \@pascal_row63,  \@pascal_row64,  \@pascal_row65,
                           \@pascal_row66,  \@pascal_row67,  \@pascal_row68,  \@pascal_row69,  \@pascal_row70,
                           \@pascal_row71,  \@pascal_row72,  \@pascal_row73,  \@pascal_row74,  \@pascal_row75,
                           \@pascal_row76,  \@pascal_row77,  \@pascal_row78,  \@pascal_row79,  \@pascal_row80,
                           \@pascal_row81,  \@pascal_row82,  \@pascal_row83,  \@pascal_row84,  \@pascal_row85,
                           \@pascal_row86,  \@pascal_row87,  \@pascal_row88,  \@pascal_row89,  \@pascal_row90,
                           \@pascal_row91,  \@pascal_row92,  \@pascal_row93,  \@pascal_row94,  \@pascal_row95,
                           \@pascal_row96,  \@pascal_row97,  \@pascal_row98,  \@pascal_row99,  \@pascal_row100,
                           \@pascal_row101, \@pascal_row102, \@pascal_row103, \@pascal_row104, \@pascal_row105,
                           \@pascal_row106, \@pascal_row107, \@pascal_row108, \@pascal_row109, \@pascal_row110,
                           \@pascal_row111, \@pascal_row112, \@pascal_row113, \@pascal_row114, \@pascal_row115,
                           \@pascal_row116, \@pascal_row117, \@pascal_row118, \@pascal_row119, \@pascal_row120,
                           \@pascal_row121, \@pascal_row122, \@pascal_row123, \@pascal_row124, \@pascal_row125,
                           \@pascal_row126, \@pascal_row127, \@pascal_row128, \@pascal_row129, \@pascal_row130,
                           \@pascal_row131, \@pascal_row132, \@pascal_row133, \@pascal_row134, \@pascal_row135,
                           \@pascal_row136, \@pascal_row137, \@pascal_row138, \@pascal_row139, \@pascal_row140,
                           \@pascal_row141, \@pascal_row142, \@pascal_row143, \@pascal_row144, \@pascal_row145,
                           \@pascal_row146, \@pascal_row147, \@pascal_row148, \@pascal_row149, \@pascal_row150,
                );

                for (my $interval = 0 ; $interval < $highest_mz ; $interval = $interval + $interval_range) {

                    my $best_score          = 1;
                    my $best_d0_score       = 1;
                    my $best_d2_score       = 1;
                    my $best_score_alpha    = 1;
                    my $best_score_beta     = 1;
                    my $best_score_alpha_d2 = 1;
                    my $best_score_beta_d2  = 1;

                    my $count = $dbh->prepare(
"SELECT COUNT(theoretical.mass)  FROM theoretical WHERE x=? AND y=? AND sequence=? AND mass > ? AND mass <= ? AND heavy_light = '0' AND is_scored = '1'"
                    );
                    _retry 15, sub {
                        $count->execute($xlink_position[0], $xlink_position[1], $sequence, $interval,
                                        $interval + $interval_range);
                    };
                    my $n = $count->fetchrow_array;

                    $count = $dbh->prepare(
"SELECT COUNT(theoretical.mass)  FROM theoretical WHERE x=? AND y=? AND sequence=? AND mass > ? AND mass <= ? AND heavy_light = '1' AND is_scored = '1'"
                    );
                    _retry 15, sub {
                        $count->execute($xlink_position[0], $xlink_position[1], $sequence, $interval,
                                        $interval + $interval_range);
                    };
                    my $d2_n = $count->fetchrow_array;

                    # 	      warn "Fast mode? $fast_mode";
                    if (!defined $fast_mode) { $fast_mode = 1 }
                    my $n_alpha;
                    my $n_alpha_d2;
                    my $n_beta;
                    my $n_beta_d2;

                    if ($fast_mode == 0) {
                        $count = $dbh->prepare(
"SELECT COUNT(theoretical.mass)  FROM theoretical WHERE x=? AND y=? AND sequence=? AND mass > ? AND mass <= ? AND heavy_light = '0' AND peptide_chain ='0'  AND is_scored = '1'"
                        );
                        _retry 15, sub {
                            $count->execute($xlink_position[0], $xlink_position[1], $sequence, $interval,
                                            $interval + $interval_range);
                        };
                        $n_alpha = $count->fetchrow_array;

                        $count = $dbh->prepare(
"SELECT COUNT(theoretical.mass)  FROM theoretical WHERE x=? AND y=? AND sequence=? AND mass > ? AND mass <= ? AND heavy_light = '1' AND peptide_chain ='0'  AND is_scored = '1'"
                        );
                        _retry 15, sub {
                            $count->execute($xlink_position[0], $xlink_position[1], $sequence, $interval,
                                            $interval + $interval_range);
                        };
                        $n_alpha_d2 = $count->fetchrow_array;

                        $count = $dbh->prepare(
"SELECT COUNT(theoretical.mass)  FROM theoretical WHERE x=? AND y=? AND sequence=? AND mass > ? AND mass <= ? AND heavy_light = '0' AND peptide_chain ='1' AND is_scored = '1'"
                        );
                        _retry 15, sub {
                            $count->execute($xlink_position[0], $xlink_position[1], $sequence, $interval,
                                            $interval + $interval_range);
                        };
                        $n_beta = $count->fetchrow_array;

                        $count = $dbh->prepare(
"SELECT COUNT(theoretical.mass)  FROM theoretical WHERE x=? AND y=? AND sequence=? AND mass > ? AND mass <= ? AND heavy_light = '1' AND peptide_chain ='1' AND is_scored = '1'"
                        );
                        _retry 15, sub {
                            $count->execute($xlink_position[0], $xlink_position[1], $sequence, $interval,
                                            $interval + $interval_range);
                        };
                        $n_beta_d2 = $count->fetchrow_array;

                    }

                    for (my $q = 1 ; $q <= $q_max ; $q++) {

# AND ms2.possible_ion_shift >= theoretical.crosslink_ion AND ms2.possible_no_ion_shift >= theoretical.not_crosslink_ion

                        my $count = $dbh->prepare(
"SELECT COUNT(DISTINCT theoretical.mass)  FROM theoretical inner join  (select  * from ms2 WHERE mass > ? AND mass <= ?  AND ms2.abundance > ? AND ms2.heavy_light = 0 ORDER BY ms2.abundance+0 DESC LIMIT ?) as top_ms2 ON (top_ms2.mass between theoretical.mass -  ? and theoretical.mass + ? ) WHERE x=? AND y=? and sequence=? AND theoretical.mass > ? AND theoretical.mass <= ? AND theoretical.heavy_light = '0' AND theoretical.is_scored = '1' AND top_ms2.possible_ion_shift >= theoretical.crosslink_ion AND top_ms2.possible_no_ion_shift >= theoretical.not_crosslink_ion  "

# "SELECT COUNT(DISTINCT theoretical.mass)  FROM theoretical inner join  (select  * from ms2 WHERE mass > ? AND mass <= ?  AND ms2.abundance > ? AND ms2.heavy_light = 0 ORDER BY ms2.abundance+0 DESC LIMIT ?) as top_ms2 ON (top_ms2.mass between theoretical.mass -  ? and theoretical.mass + ? ) WHERE x=? AND y=? and sequence=? AND theoretical.mass > ? AND theoretical.mass <= ? AND theoretical.heavy_light = '0' AND theoretical.is_scored = '1'  "

                        );
                        _retry 15, sub {
                            $count->execute(
                                            $interval,                           $interval + $interval_range,
                                            ($max_abundance * $threshold / 100), $q,
                                            $ms2_error,                          $ms2_error,
                                            $xlink_position[0],                  $xlink_position[1],
                                            $sequence,                           $interval,
                                            $interval + $interval_range
                            );
                        };
                        my $k = $count->fetchrow_array;

                        $count = $dbh->prepare(
"SELECT COUNT(DISTINCT theoretical.mass)  FROM theoretical inner join (select  * from ms2 WHERE mass > ? AND mass <= ?  AND ms2.abundance > ? AND ms2.heavy_light = 1  ORDER BY ms2.abundance+0 DESC LIMIT ?) as top_ms2 on (top_ms2.mass between theoretical.mass -  ? and theoretical.mass + ?) WHERE x=? AND y=? and sequence=? AND theoretical.mass > ? AND theoretical.mass <= ? AND theoretical.heavy_light = '1' AND theoretical.is_scored = '1' AND top_ms2.possible_ion_shift >= theoretical.crosslink_ion AND top_ms2.possible_no_ion_shift >= theoretical.not_crosslink_ion  "
                        );
                        _retry 15, sub {
                            $count->execute(
                                            $interval,                              $interval + $interval_range,
                                            ($max_abundance_d2 * $threshold / 100), $q,
                                            $ms2_error,                             $ms2_error,
                                            $xlink_position[0],                     $xlink_position[1],
                                            $sequence,                              $interval,
                                            $interval + $interval_range
                            );
                        };
                        my $d2_k = $count->fetchrow_array;

                        my $k_alpha;
                        my $k_alpha_d2;
                        my $k_beta;
                        my $k_beta_d2;

                        if ($fast_mode == 0) {
                            $count = $dbh->prepare(
"SELECT COUNT(DISTINCT theoretical.mass)  FROM theoretical inner join (select  * from ms2 WHERE mass > ? AND mass <= ?  AND ms2.abundance > ?  AND ms2.heavy_light = 0  ORDER BY ms2.abundance+0 DESC LIMIT ?) as top_ms2 on (top_ms2.mass between theoretical.mass -  ? and theoretical.mass + ?) WHERE x=? AND y=? and sequence=? AND theoretical.mass > ? AND theoretical.mass <= ? AND theoretical.heavy_light = '0' AND theoretical.is_scored = '1'AND theoretical.peptide_chain = '0' AND top_ms2.possible_ion_shift >= theoretical.crosslink_ion AND top_ms2.possible_no_ion_shift >= theoretical.not_crosslink_ion  "
                            );
                            _retry 15, sub {
                                $count->execute(
                                                $interval,                           $interval + $interval_range,
                                                ($max_abundance * $threshold / 100), $q,
                                                $ms2_error,                          $ms2_error,
                                                $xlink_position[0],                  $xlink_position[1],
                                                $sequence,                           $interval,
                                                $interval + $interval_range
                                );
                            };
                            $k_alpha = $count->fetchrow_array;

                            $count = $dbh->prepare(
"SELECT COUNT(DISTINCT theoretical.mass)  FROM theoretical inner join (select  * from ms2 WHERE mass > ? AND mass <= ?  AND ms2.abundance > ? AND ms2.heavy_light = 1  ORDER BY ms2.abundance+0 DESC LIMIT ?) as top_ms2 on (top_ms2.mass between theoretical.mass -  ? and theoretical.mass + ?) WHERE x=? AND y=? and sequence=? AND theoretical.mass > ? AND theoretical.mass <= ? AND theoretical.heavy_light = '1' AND theoretical.is_scored = '1' and theoretical.peptide_chain = '0' AND top_ms2.possible_ion_shift >= theoretical.crosslink_ion AND top_ms2.possible_no_ion_shift >= theoretical.not_crosslink_ion  "
                            );
                            _retry 15, sub {
                                $count->execute(
                                                $interval,                              $interval + $interval_range,
                                                ($max_abundance_d2 * $threshold / 100), $q,
                                                $ms2_error,                             $ms2_error,
                                                $xlink_position[0],                     $xlink_position[1],
                                                $sequence,                              $interval,
                                                $interval + $interval_range
                                );
                            };
                            $k_alpha_d2 = $count->fetchrow_array;

                            $count = $dbh->prepare(
"SELECT COUNT(DISTINCT theoretical.mass)  FROM theoretical inner join (select  * from ms2 WHERE mass > ? AND mass <= ?  AND ms2.abundance > ?  AND ms2.heavy_light = 0  ORDER BY ms2.abundance+0 DESC LIMIT ?) as top_ms2 on (top_ms2.mass between theoretical.mass -  ? and theoretical.mass + ?) WHERE x=? AND y=? and sequence=? AND theoretical.mass > ? AND theoretical.mass <= ? AND theoretical.heavy_light = '0' AND theoretical.is_scored = '1' AND theoretical.peptide_chain = '1' AND top_ms2.possible_ion_shift >= theoretical.crosslink_ion AND top_ms2.possible_no_ion_shift >= theoretical.not_crosslink_ion  "
                            );
                            _retry 15, sub {
                                $count->execute(
                                                $interval,                           $interval + $interval_range,
                                                ($max_abundance * $threshold / 100), $q,
                                                $ms2_error,                          $ms2_error,
                                                $xlink_position[0],                  $xlink_position[1],
                                                $sequence,                           $interval,
                                                $interval + $interval_range
                                );
                            };
                            $k_beta = $count->fetchrow_array;

                            $count = $dbh->prepare(
"SELECT COUNT(DISTINCT theoretical.mass)  FROM theoretical inner join (select  * from ms2 WHERE mass > ? AND mass <= ?  AND ms2.abundance > ?  AND ms2.heavy_light = 1  ORDER BY ms2.abundance+0 DESC LIMIT ?) as top_ms2 on (top_ms2.mass between theoretical.mass -  ? and theoretical.mass + ?) WHERE x=? AND y=? and sequence=? AND theoretical.mass > ? AND theoretical.mass <= ? AND theoretical.heavy_light = '1' AND theoretical.is_scored = '1' AND theoretical.peptide_chain = '1' AND top_ms2.possible_ion_shift >= theoretical.crosslink_ion AND top_ms2.possible_no_ion_shift >= theoretical.not_crosslink_ion  "
                            );
                            _retry 15, sub {
                                $count->execute(
                                                $interval,                           $interval + $interval_range,
                                                ($max_abundance * $threshold / 100), $q,
                                                $ms2_error,                          $ms2_error,
                                                $xlink_position[0],                  $xlink_position[1],
                                                $sequence,                           $interval,
                                                $interval + $interval_range
                                );
                            };
                            $k_beta_d2 = $count->fetchrow_array;
                        }

                        my $d0_score       = 0;
                        my $d2_score       = 0;
                        my $score_alpha    = 1;
                        my $score_beta     = 1;
                        my $score_alpha_d2 = 1;
                        my $score_beta_d2  = 1;

                        for (my $j = $k ; $j <= $n ; $j++) {
                            my $binomial_coefficient;
                            if ($n > 150) {
                                my $binomial = Math::BigRat->new(scalar($n));
                                $binomial->bnok($j);
                                $binomial_coefficient = $binomial->as_float();

                                #    warn "ncr=$nCr[$n-1][$j]  bigrat=$binomial_coefficient";
                            } else {
                                $binomial_coefficient = $nCr[ $n - 1 ][$j];
                            }
                            $d0_score = $d0_score +
                              (scalar $binomial_coefficient * (($q / 100)**$j) * ((1 - ($q / 100))**($n - $j)));

                        }

                        for (my $j = $d2_k ; $j <= $d2_n ; $j++) {
                            my $binomial_coefficient;
                            if ($d2_n > 150) {
                                my $binomial = Math::BigRat->new(scalar($d2_n));
                                $binomial->bnok($j);
                                $binomial_coefficient = $binomial->as_float();
                            } else {
                                $binomial_coefficient = $nCr[ $d2_n - 1 ][$j];
                            }
                            $d2_score = $d2_score +
                              (scalar $binomial_coefficient * (($q / 100)**$j) * ((1 - ($q / 100))**($d2_n - $j)));

                        }

                        if ($fast_mode == 0) {
                            $score_alpha    = 0;
                            $score_beta     = 0;
                            $score_alpha_d2 = 0;
                            $score_beta_d2  = 0;
                            for (my $j = $k_alpha ; $j <= $n_alpha ; $j++) {
                                my $binomial_coefficient;
                                if ($n_alpha > 150) {
                                    my $binomial = Math::BigRat->new(scalar($n_alpha));
                                    $binomial->bnok($j);
                                    $binomial_coefficient = $binomial->as_float();

                                   #                            warn "ncr=$nCr[$n-1][$j]  bigrat=$binomial_coefficient";
                                } else {
                                    $binomial_coefficient = $nCr[ $n_alpha - 1 ][$j];
                                }
                                $score_alpha =
                                  $score_alpha +
                                  (
                                    scalar $binomial_coefficient *
                                      (($q / 100)**$j) *
                                      ((1 - ($q / 100))**($n_alpha - $j)));

                         #  		     warn 'Alpha:',$score_alpha, ",k=" ,$k_alpha, "/$n_alpha($n_beta)/$n," , $q, "," , $j;
                            }

                            for (my $j = $k_alpha_d2 ; $j <= $n_alpha_d2 ; $j++) {
                                my $binomial_coefficient;
                                if ($n_alpha_d2 > 150) {
                                    my $binomial = Math::BigRat->new(scalar($n_alpha_d2));
                                    $binomial->bnok($j);
                                    $binomial_coefficient = $binomial->as_float();

                                   #                            warn "ncr=$nCr[$n-1][$j]  bigrat=$binomial_coefficient";
                                } else {
                                    $binomial_coefficient = $nCr[ $n_alpha_d2 - 1 ][$j];
                                }
                                $score_alpha_d2 =
                                  $score_alpha_d2 +
                                  (
                                    scalar $binomial_coefficient *
                                      (($q / 100)**$j) *
                                      ((1 - ($q / 100))**($n_alpha_d2 - $j)));

                         #  		     warn 'Alpha:',$score_alpha, ",k=" ,$k_alpha, "/$n_alpha($n_beta)/$n," , $q, "," , $j;
                            }

                            for (my $j = $k_beta ; $j <= $n_beta ; $j++) {
                                my $binomial_coefficient;
                                if ($n_beta > 150) {
                                    my $binomial = Math::BigRat->new(scalar($n_beta_d2));
                                    $binomial->bnok($j);
                                    $binomial_coefficient = $binomial->as_float();

                                    #    warn "ncr=$nCr[$n-1][$j]  bigrat=$binomial_coefficient";
                                } else {
                                    $binomial_coefficient = $nCr[ $n_beta - 1 ][$j];
                                }
                                $score_beta =
                                  $score_beta +
                                  (
                                    scalar $binomial_coefficient * (($q / 100)**$j) * ((1 - ($q / 100))**($n_beta - $j))
                                  );

                            }

                            for (my $j = $k_beta_d2 ; $j <= $n_beta_d2 ; $j++) {
                                my $binomial_coefficient;
                                if ($n_beta_d2 > 150) {
                                    my $binomial = Math::BigRat->new(scalar($n_beta));
                                    $binomial->bnok($j);
                                    $binomial_coefficient = $binomial->as_float();

                                    #    warn "ncr=$nCr[$n-1][$j]  bigrat=$binomial_coefficient";
                                } else {
                                    $binomial_coefficient = $nCr[ $n_beta_d2 - 1 ][$j];
                                }
                                $score_beta_d2 =
                                  $score_beta_d2 +
                                  (
                                    scalar $binomial_coefficient *
                                      (($q / 100)**$j) *
                                      ((1 - ($q / 100))**($n_beta_d2 - $j)));

                            }

                        }

                        my $score = $d0_score * $d2_score;

                        if ($best_score > $score) {
                            $best_score          = $score;
                            $best_d0_score       = $d0_score;
                            $best_d2_score       = $d2_score;
                            $best_score_alpha    = $score_alpha;
                            $best_score_beta     = $score_beta;
                            $best_score_alpha_d2 = $score_alpha_d2;
                            $best_score_beta_d2  = $score_beta_d2;
                        }

                    }

                    $d0_score_total       = $best_d0_score * $d0_score_total;
                    $d2_score_total       = $best_d2_score * $d2_score_total;
                    $score_alpha_total    = $best_score_alpha * $score_alpha_total;
                    $score_beta_total     = $best_score_beta * $score_beta_total;
                    $score_alpha_d2_total = $best_score_alpha_d2 * $score_alpha_d2_total;
                    $score_beta_d2_total  = $best_score_beta_d2 * $score_beta_d2_total;
                    $score_total          = $best_score * $score_total;

                }

                if ((-10 * log $score_total) > $best_match) {

# 	       warn "d0_score = $d0_score_total,d2_score = $d2_score_total, alpha_score=$score_alpha_total, alpha_d2_score=$score_alpha_d2_total";
                    $best_match    = -10 * log $score_total;
                    $best_alpha    = -10 * log($score_alpha_total * $score_alpha_d2_total);
                    $best_beta     = -10 * log($score_beta_total * $score_beta_d2_total);
                    $best_sequence = $sequence;
                    $best_x        = $xlink_pos[0][$x];
                    $best_y        = $xlink_pos[1][$y];
                }
            }

        }
    }

#    my $matchlist = $dbh->prepare("SELECT *,  ms2.abundance as abundance FROM theoretical inner join ms2 on (ms2.mass between theoretical.mass -  ? and theoretical.mass + ? ) WHERE x=? AND y=? and sequence=? AND ms2.heavy_light = '0' and theoretical.heavy_light = '0' AND abundance > ?  ORDER by abundance DESC");
    my $matchlist = $dbh->prepare(
"SELECT *,  ms2.abundance as abundance FROM theoretical inner join ms2 on (ms2.mass between theoretical.mass -  ? and theoretical.mass + ? ) WHERE x=? AND y=? and sequence=? AND ms2.heavy_light = '0' and theoretical.heavy_light = '0'  AND abundance > ? AND ms2.possible_ion_shift >= theoretical.crosslink_ion AND ms2.possible_no_ion_shift >= theoretical.not_crosslink_ion  ORDER by abundance DESC"
    );

    _retry 15, sub {
        $matchlist->execute($ms2_error, $ms2_error, $best_x, $best_y, $best_sequence,
                            ($max_abundance * $threshold / 100));
    };
    my $chain;
    my $top_10;
    while (my $peaks = $matchlist->fetchrow_hashref) {
        if ($peaks->{peptide_chain} == 0) {
            $chain = "&#945;";
        } else {
            $chain = "&#946;";
        }
        $top_10 = $top_10
          . "$chain$peaks->{ion_type}$peaks->{mod}<sub>$peaks->{position}</sub><sup>$peaks->{charge}+</sup> = $peaks->{mass}<br/>\n";
    }

    my $matched_abundance = 0;
    my $matched_common    = 0;
    my $matched_crosslink = 0;
    $matchlist = $dbh->prepare(
"SELECT *,  ms2.abundance as abundance FROM theoretical inner join ms2 on (ms2.mass between theoretical.mass -  ? and theoretical.mass + ? ) WHERE x=? AND y=? and sequence=? AND ms2.heavy_light = '0' and theoretical.heavy_light = '0' AND abundance > ? GROUP BY ms2.mass"
    );
    _retry 15, sub {
        $matchlist->execute($ms2_error, $ms2_error, $best_x, $best_y, $best_sequence,
                            ($max_abundance * $threshold / 100));
    };
    while (my $peaks = $matchlist->fetchrow_hashref) {
        if ($peaks->{crosslink_ion} == 0) {
            $matched_common = $matched_common + 1;
        } else {
            $matched_crosslink = $matched_crosslink + 1;
        }
        $matched_abundance = $matched_abundance + $peaks->{abundance};
    }

    $matchlist = $dbh->prepare("SELECT *  FROM  ms2  WHERE ms2.heavy_light = '0'");
    _retry 15, sub { $matchlist->execute() };
    my $total_abundance = 0;
    while (my $peaks = $matchlist->fetchrow_hashref) {
        $total_abundance = $total_abundance + $peaks->{abundance};
    }

    $matchlist = $dbh->prepare(
"SELECT *,  ms2.abundance as abundance FROM theoretical inner join ms2 on (ms2.mass between theoretical.mass -  ? and theoretical.mass + ?) WHERE x=? AND y=? and sequence=? AND ms2.heavy_light = '1' and theoretical.heavy_light = '1' AND abundance > ? AND ms2.possible_ion_shift >= theoretical.crosslink_ion AND ms2.possible_no_ion_shift >= theoretical.not_crosslink_ion    ORDER by abundance DESC "
    );
    _retry 15, sub {
        $matchlist->execute($ms2_error, $ms2_error, $best_x, $best_y, $best_sequence,
                            ($max_abundance_d2 * $threshold / 100));
    };

    my $d2_top_10 = '';
    while (my $peaks = $matchlist->fetchrow_hashref) {
        if ($peaks->{peptide_chain} == 0) {
            $chain = "&#945;";
        } else {
            $chain = "&#946;";
        }
        $d2_top_10 = $d2_top_10
          . "$chain$peaks->{ion_type}$peaks->{mod}<sub>$peaks->{position}</sub><sup>$peaks->{charge}+</sup> = $peaks->{mass}<br/>\n";
    }

    my $d2_matched_abundance = 0;
    my $d2_matched_common    = 0;
    my $d2_matched_crosslink = 0;
    $matchlist = $dbh->prepare(
"SELECT *,  ms2.abundance as abundance FROM theoretical inner join ms2 on (ms2.mass between theoretical.mass -  ? and theoretical.mass + ? ) WHERE x=? AND y=? and sequence=? AND ms2.heavy_light = '1' and theoretical.heavy_light = '1' AND abundance > ? GROUP BY ms2.mass"
    );
    _retry 15, sub {
        $matchlist->execute($ms2_error, $ms2_error, $best_x, $best_y, $best_sequence,
                            ($max_abundance_d2 * $threshold / 100));
    };
    while (my $peaks = $matchlist->fetchrow_hashref) {
        if ($peaks->{crosslink_ion} == 0) {
            $d2_matched_common = $d2_matched_common + 1;
        } else {
            $d2_matched_crosslink = $d2_matched_crosslink + 1;
        }
        $d2_matched_abundance = $d2_matched_abundance + $peaks->{abundance};
    }

    $matchlist = $dbh->prepare("SELECT *  FROM  ms2  WHERE ms2.heavy_light = '1'");
    _retry 15, sub { $matchlist->execute() };
    my $d2_total_abundance = 0;
    while (my $peaks = $matchlist->fetchrow_hashref) {
        $d2_total_abundance = $d2_total_abundance + $peaks->{abundance};
    }

    # warn $d2_top_10;

    $matchlist->finish();
    $dbh->disconnect();

    #     if ( $best_match > 0 ) {
    #        warn "Best: $best_sequence - Score: ", sprintf( "%.0f", ($best_match) ),"\n";
    #     }
    my $td = tv_interval($t0);

    #     warn "Time Taken = ", $td, " secs \n";
    my $returned_score = sprintf("%.0f", ($best_match));
    $best_alpha = sprintf("%.0f", ($best_alpha));
    $best_beta  = sprintf("%.0f", ($best_beta));

    my $min_chain_score;
    if   ($best_alpha < $best_beta) { $min_chain_score = $best_alpha }
    else                            { $min_chain_score = $best_beta }
    return (
            $returned_score,    $best_sequence,        $best_x,            $best_y,
            $top_10,            $d2_top_10,            $matched_abundance, $d2_matched_abundance,
            $total_abundance,   $d2_total_abundance,   $matched_common,    $matched_crosslink,
            $d2_matched_common, $d2_matched_crosslink, $best_alpha,        $best_beta,
            $min_chain_score
    );
}

sub find_ion_series {

    my ($top_10) = @_;
    my @ion_series = $top_10 =~ m/\&\#[0-9].*\;[A-Z]<sub>[0-9].*<\/sub><sup>[0-9]\+<\/sup>/g;
    my %ion_series_sorted;

    foreach my $ion (sort @ion_series) {
        $ion =~ /\&\#([0-9].*)\;([A-Z])<sub>([0-9].*)<\/sub><sup>([0-9])\+<\/sup>/;
        $ion_series_sorted{"$1,$2,$4,$3"} =
          1 #using a hash to avoid duplicates, if a double match occurs for some reason (it did early on, but shouldn't anymore).
    }

    my $current_chain        = '';
    my $current_ion_type     = '';
    my $current_charge_state = '';
    my $last_fragment_number = -1;
    my $ion_series_length    = 0;
    my %max_ion_series_length;
    $max_ion_series_length{'total'}   = 0;
    $max_ion_series_length{'alpha_b'} = 0;
    $max_ion_series_length{'alpha_y'} = 0;
    $max_ion_series_length{'beta_b'}  = 0;
    $max_ion_series_length{'beta_y'}  = 0;

    foreach my $ion (sort keys %ion_series_sorted) {
        my ($chain, $ion_type, $charge_state, $fragment_number) = split(",", $ion);

        if ($ion_series_length > $max_ion_series_length{'total'}) {
            $max_ion_series_length{'total'} = $ion_series_length;
        }

        if ($current_chain eq '945' && $current_ion_type eq 'B') {
            if ($ion_series_length > $max_ion_series_length{'alpha_b'}) {
                $max_ion_series_length{'alpha_b'} = $ion_series_length;
            }
        } elsif ($current_chain eq '945' && $current_ion_type eq 'Y') {
            if ($ion_series_length > $max_ion_series_length{'alpha_y'}) {
                $max_ion_series_length{'alpha_y'} = $ion_series_length;
            }
        } elsif ($current_chain eq '946' && $current_ion_type eq 'B') {
            if ($ion_series_length > $max_ion_series_length{'beta_b'}) {
                $max_ion_series_length{'beta_b'} = $ion_series_length;
            }
        } elsif ($current_chain eq '946' && $current_ion_type eq 'Y') {
            if ($ion_series_length > $max_ion_series_length{'beta_y'}) {
                $max_ion_series_length{'beta_y'} = $ion_series_length;
            }
        }

        if ($chain ne $current_chain) {
            $current_chain        = $chain;
            $ion_series_length    = 0;
            $last_fragment_number = -1;
        }
        if ($ion_type ne $current_ion_type) {
            $current_ion_type     = $ion_type;
            $ion_series_length    = 0;
            $last_fragment_number = -1;
        }
        if ($charge_state ne $current_charge_state) {
            $current_charge_state = $charge_state;
            $ion_series_length    = 0;
            $last_fragment_number = -1;
        }
        if ($fragment_number == $last_fragment_number + 1) {

            $ion_series_length    = $ion_series_length + 1;
            $last_fragment_number = $fragment_number;
        } else {
            $ion_series_length = 0;
        }

#  print "$ion  -  $ion_series_length - $max_ion_series_length{'total'}, aB $max_ion_series_length{'alpha_b'} aY $max_ion_series_length{'alpha_y'} bB $max_ion_series_length{'beta_b'} bY $max_ion_series_length{'beta_y'} <br>";
        $last_fragment_number = $fragment_number;
    }

    return \%max_ion_series_length;

}

1;
